import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { AlertCircle, Play, Pause, Square, CheckCircle2, Clock, User, Monitor } from "lucide-react";
import { DashboardLayout } from "@/components/dashboard-layout";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";

interface SystemOperation {
  id: number;
  userId: number;
  channelId: number;
  videoId?: number;
  actionType: string;
  status: string;
  currentStep?: string;
  totalSteps?: number;
  progress: number;
  createdAt: string;
  updatedAt: string;
  completedAt?: string;
  user?: {
    id: number;
    username: string;
    email: string;
  };
  channel?: {
    id: number;
    channelName: string;
  };
  videoTitle?: string;
}

export default function SystemOperations() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [autoRefresh, setAutoRefresh] = useState(true);

  // Fetch all system operations
  const { data: operations = [], isLoading, refetch } = useQuery({
    queryKey: ["/api/admin/system-operations"],
    refetchInterval: autoRefresh ? 5000 : false, // Auto-refresh every 5 seconds
  });

  // Force complete operation
  const forceCompleteMutation = useMutation({
    mutationFn: async (itemId: number) => {
      return apiRequest(`/api/admin/queue/${itemId}/force-complete`, "POST");
    },
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Operation force completed successfully",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/system-operations"] });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: "Failed to force complete operation",
        variant: "destructive",
      });
    },
  });

  // Pause operation
  const pauseMutation = useMutation({
    mutationFn: async (itemId: number) => {
      return apiRequest(`/api/admin/queue/${itemId}/pause`, "POST");
    },
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Operation paused successfully",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/system-operations"] });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: "Failed to pause operation",
        variant: "destructive",
      });
    },
  });

  // Resume operation
  const resumeMutation = useMutation({
    mutationFn: async (itemId: number) => {
      return apiRequest(`/api/admin/queue/${itemId}/resume`, "POST");
    },
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Operation resumed successfully",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/system-operations"] });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: "Failed to resume operation",
        variant: "destructive",
      });
    },
  });

  // Cancel operation
  const cancelMutation = useMutation({
    mutationFn: async (itemId: number) => {
      return apiRequest(`/api/admin/queue/${itemId}/cancel`, "POST");
    },
    onSuccess: () => {
      toast({
        title: "Success",
        description: "Operation cancelled successfully",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/system-operations"] });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: "Failed to cancel operation",
        variant: "destructive",
      });
    },
  });

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'completed':
        return <CheckCircle2 className="h-4 w-4 text-green-600" />;
      case 'running':
        return <Play className="h-4 w-4 text-blue-600" />;
      case 'paused':
        return <Pause className="h-4 w-4 text-yellow-600" />;
      case 'failed':
        return <AlertCircle className="h-4 w-4 text-red-600" />;
      default:
        return <Clock className="h-4 w-4 text-gray-600" />;
    }
  };

  const getStatusBadge = (status: string) => {
    const variants: Record<string, "default" | "secondary" | "destructive" | "outline"> = {
      completed: "default",
      running: "secondary",
      paused: "outline",
      failed: "destructive",
      queued: "outline"
    };
    return <Badge variant={variants[status] || "outline"}>{status}</Badge>;
  };

  const isOperationStuck = (operation: SystemOperation) => {
    const updatedAt = new Date(operation.updatedAt);
    const now = new Date();
    const timeDiff = now.getTime() - updatedAt.getTime();
    const minutesDiff = timeDiff / (1000 * 60);
    
    // Consider stuck if running for more than 10 minutes without updates
    return operation.status === 'running' && minutesDiff > 10;
  };

  const activeOperations = operations.filter((op: SystemOperation) => 
    ['running', 'paused', 'queued'].includes(op.status)
  );
  
  const completedOperations = operations.filter((op: SystemOperation) => 
    ['completed', 'failed', 'cancelled'].includes(op.status)
  );

  const stuckOperations = operations.filter((op: SystemOperation) => isOperationStuck(op));

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
          <div>
            <h1 className="text-2xl sm:text-3xl font-bold text-gray-900">System Operations</h1>
            <p className="text-gray-600 mt-1 sm:mt-2 text-sm sm:text-base">
              Monitor all system operations across all users and channels
            </p>
          </div>
          
          <div className="flex items-center space-x-2">
            <Button
              onClick={() => setAutoRefresh(!autoRefresh)}
              variant={autoRefresh ? "default" : "outline"}
              size="sm"
            >
              <Monitor className="h-4 w-4 mr-2" />
              {autoRefresh ? "Live" : "Paused"}
            </Button>
            <Button onClick={() => refetch()} variant="outline" size="sm">
              Refresh
            </Button>
          </div>
        </div>

        {/* Statistics Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Total Operations</CardTitle>
              <Monitor className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{operations.length}</div>
            </CardContent>
          </Card>
          
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Active Operations</CardTitle>
              <Play className="h-4 w-4 text-blue-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-blue-600">{activeOperations.length}</div>
            </CardContent>
          </Card>
          
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Stuck Operations</CardTitle>
              <AlertCircle className="h-4 w-4 text-red-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-red-600">{stuckOperations.length}</div>
            </CardContent>
          </Card>
          
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Completed Today</CardTitle>
              <CheckCircle2 className="h-4 w-4 text-green-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-green-600">
                {completedOperations.filter(op => {
                  const today = new Date().toDateString();
                  const opDate = new Date(op.completedAt || op.updatedAt).toDateString();
                  return today === opDate;
                }).length}
              </div>
            </CardContent>
          </Card>
        </div>

        <Tabs defaultValue="active" className="space-y-4">
          <TabsList>
            <TabsTrigger value="active">Active Operations ({activeOperations.length})</TabsTrigger>
            <TabsTrigger value="stuck">Stuck Operations ({stuckOperations.length})</TabsTrigger>
            <TabsTrigger value="completed">Completed ({completedOperations.length})</TabsTrigger>
            <TabsTrigger value="all">All Operations ({operations.length})</TabsTrigger>
          </TabsList>

          <TabsContent value="active" className="space-y-4">
            {activeOperations.length === 0 ? (
              <Card>
                <CardContent className="flex items-center justify-center h-32">
                  <p className="text-gray-500">No active operations</p>
                </CardContent>
              </Card>
            ) : (
              activeOperations.map((operation: SystemOperation) => (
                <OperationCard 
                  key={operation.id} 
                  operation={operation}
                  onForceComplete={() => forceCompleteMutation.mutate(operation.id)}
                  onPause={() => pauseMutation.mutate(operation.id)}
                  onResume={() => resumeMutation.mutate(operation.id)}
                  onCancel={() => cancelMutation.mutate(operation.id)}
                />
              ))
            )}
          </TabsContent>

          <TabsContent value="stuck" className="space-y-4">
            {stuckOperations.length === 0 ? (
              <Card>
                <CardContent className="flex items-center justify-center h-32">
                  <p className="text-gray-500">No stuck operations</p>
                </CardContent>
              </Card>
            ) : (
              stuckOperations.map((operation: SystemOperation) => (
                <OperationCard 
                  key={operation.id} 
                  operation={operation}
                  isStuck={true}
                  onForceComplete={() => forceCompleteMutation.mutate(operation.id)}
                  onPause={() => pauseMutation.mutate(operation.id)}
                  onResume={() => resumeMutation.mutate(operation.id)}
                  onCancel={() => cancelMutation.mutate(operation.id)}
                />
              ))
            )}
          </TabsContent>

          <TabsContent value="completed" className="space-y-4">
            {completedOperations.slice(0, 50).map((operation: SystemOperation) => (
              <OperationCard key={operation.id} operation={operation} />
            ))}
          </TabsContent>

          <TabsContent value="all" className="space-y-4">
            {operations.slice(0, 100).map((operation: SystemOperation) => (
              <OperationCard key={operation.id} operation={operation} />
            ))}
          </TabsContent>
        </Tabs>
      </div>
    </DashboardLayout>
  );
}

interface OperationCardProps {
  operation: SystemOperation;
  isStuck?: boolean;
  onForceComplete?: () => void;
  onPause?: () => void;
  onResume?: () => void;
  onCancel?: () => void;
}

function OperationCard({ 
  operation, 
  isStuck = false, 
  onForceComplete, 
  onPause, 
  onResume, 
  onCancel 
}: OperationCardProps) {
  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'completed':
        return <CheckCircle2 className="h-4 w-4 text-green-600" />;
      case 'running':
        return <Play className="h-4 w-4 text-blue-600" />;
      case 'paused':
        return <Pause className="h-4 w-4 text-yellow-600" />;
      case 'failed':
        return <AlertCircle className="h-4 w-4 text-red-600" />;
      default:
        return <Clock className="h-4 w-4 text-gray-600" />;
    }
  };

  const getStatusBadge = (status: string) => {
    const variants: Record<string, "default" | "secondary" | "destructive" | "outline"> = {
      completed: "default",
      running: "secondary",
      paused: "outline",
      failed: "destructive",
      queued: "outline"
    };
    return <Badge variant={variants[status] || "outline"}>{status}</Badge>;
  };

  return (
    <Card className={isStuck ? "border-red-200 bg-red-50" : ""}>
      <CardHeader>
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-2">
            {getStatusIcon(operation.status)}
            <CardTitle className="text-lg">
              {operation.actionType.charAt(0).toUpperCase() + operation.actionType.slice(1)} Operation
            </CardTitle>
            {getStatusBadge(operation.status)}
            {isStuck && <Badge variant="destructive">STUCK</Badge>}
          </div>
          <div className="text-sm text-gray-500">
            ID: {operation.id}
          </div>
        </div>
        <CardDescription>
          <div className="flex items-center space-x-4 text-sm">
            <div className="flex items-center space-x-1">
              <User className="h-4 w-4" />
              <span>{operation.user?.username || 'Unknown User'}</span>
            </div>
            <div>
              Channel: {operation.channel?.channelName || 'Unknown Channel'}
            </div>
            {operation.videoTitle && (
              <div>
                Video: {operation.videoTitle}
              </div>
            )}
          </div>
        </CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {operation.status === 'running' && (
            <div className="space-y-2">
              <div className="flex justify-between text-sm">
                <span>Progress</span>
                <span>{operation.progress}%</span>
              </div>
              <Progress value={operation.progress} className="w-full" />
              {operation.currentStep && (
                <p className="text-sm text-gray-600">
                  Step {operation.currentStep}/{operation.totalSteps}: Current operation in progress
                </p>
              )}
            </div>
          )}
          
          <div className="grid grid-cols-2 gap-4 text-sm">
            <div>
              <span className="font-medium">Started:</span>
              <br />
              {new Date(operation.createdAt).toLocaleString()}
            </div>
            <div>
              <span className="font-medium">Last Updated:</span>
              <br />
              {new Date(operation.updatedAt).toLocaleString()}
            </div>
          </div>

          {(onForceComplete || onPause || onResume || onCancel) && (
            <div className="flex space-x-2 pt-4">
              {operation.status === 'running' && onPause && (
                <Button onClick={onPause} size="sm" variant="outline">
                  <Pause className="h-4 w-4 mr-1" />
                  Pause
                </Button>
              )}
              {operation.status === 'paused' && onResume && (
                <Button onClick={onResume} size="sm" variant="outline">
                  <Play className="h-4 w-4 mr-1" />
                  Resume
                </Button>
              )}
              {['running', 'paused', 'queued'].includes(operation.status) && onCancel && (
                <Button onClick={onCancel} size="sm" variant="destructive">
                  <Square className="h-4 w-4 mr-1" />
                  Cancel
                </Button>
              )}
              {(operation.status === 'running' || isStuck) && onForceComplete && (
                <Button onClick={onForceComplete} size="sm" variant="default">
                  <CheckCircle2 className="h-4 w-4 mr-1" />
                  Force Complete
                </Button>
              )}
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  );
}