import { useState, useEffect } from "react";
import { useAuth } from "@/hooks/use-auth";
import { useToast } from "@/hooks/use-toast";
import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger, DialogDescription } from "@/components/ui/dialog";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Trash2, UserPlus, Users, Shield, User, Edit, LogOut } from "lucide-react";
import { Link, useLocation } from "wouter";

interface SystemUser {
  id: number;
  username: string;
  email: string;
  role: string;
  isActive: boolean;
  lastLogin: string | null;
  createdAt: string;
}

interface UserPoints {
  remainingPoints: number;
  totalPoints: number;
  usedPoints: number;
}

export default function SystemUsers() {
  const { user, logout } = useAuth();
  const { toast } = useToast();
  const [location] = useLocation();
  const [isCreateModalOpen, setIsCreateModalOpen] = useState(false);
  const [isEditModalOpen, setIsEditModalOpen] = useState(false);
  const [editingUser, setEditingUser] = useState<SystemUser | null>(null);
  const [newUser, setNewUser] = useState({
    username: "",
    email: "",
    password: "",
    role: "client"
  });
  const [editUserData, setEditUserData] = useState({
    username: "",
    email: "",
    password: "",
    role: "client",
    isActive: true,
    pointsToAdd: ""
  });
  const queryClient = useQueryClient();

  // Fetch system users
  const { data: users = [], isLoading } = useQuery<SystemUser[]>({
    queryKey: ["/api/system-users"],
    enabled: user?.role === "admin"
  });

  // Fetch points for all users
  const { data: userPointsMap = new Map() } = useQuery<Map<number, UserPoints>>({
    queryKey: ["/api/system-users", "points"],
    enabled: user?.role === "admin" && users.length > 0,
    queryFn: async () => {
      const pointsMap = new Map<number, UserPoints>();
      
      // Fetch points for each user
      await Promise.all(
        users.map(async (systemUser) => {
          try {
            const response = await fetch(`/api/points/${systemUser.id}`);
            const pointsData = await response.json();
            pointsMap.set(systemUser.id, pointsData);
          } catch (error) {
            console.error(`Failed to fetch points for user ${systemUser.id}:`, error);
            // Set default values if fetch fails
            pointsMap.set(systemUser.id, { remainingPoints: 0, totalPoints: 0, usedPoints: 0 });
          }
        })
      );
      
      return pointsMap;
    }
  });

  // Create user mutation
  const createUserMutation = useMutation({
    mutationFn: async (userData: typeof newUser) => {
      // apiRequest already handles response parsing and error checking
      const responseData = await apiRequest("/api/system-users", "POST", userData);
      return responseData;
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/system-users"] });
      setIsCreateModalOpen(false);
      setNewUser({ username: "", email: "", password: "", role: "client" });
      toast({
        title: "User Created",
        description: `Successfully created user "${data.username}" with ${data.role} role`
      });
    },
    onError: (error: any) => {
      console.error("User creation error:", error);
      toast({
        title: "Creation Failed",
        description: error.message || "Unable to create user. Please check all fields and try again.",
        variant: "destructive"
      });
    }
  });

  // Update user mutation
  const updateUserMutation = useMutation({
    mutationFn: async ({ userId, userData }: { userId: number; userData: typeof editUserData }) => {
      // apiRequest already handles response parsing and error checking
      const responseData = await apiRequest(`/api/system-users/${userId}`, "PUT", userData);
      return responseData;
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/system-users"] });
      queryClient.invalidateQueries({ queryKey: ["/api/system-users", "points"] });
      setIsEditModalOpen(false);
      setEditingUser(null);
      setEditUserData({ username: "", email: "", password: "", role: "client", isActive: true, pointsToAdd: "" });
      
      // Enhanced success message
      let description = "User information updated successfully";
      if (editUserData.password && editUserData.password.trim() !== '') {
        description += " (password changed)";
      }
      if (editUserData.pointsToAdd && editUserData.pointsToAdd.trim() !== '') {
        description += ` (+${editUserData.pointsToAdd} points added)`;
      }
      
      toast({
        title: "Success",
        description
      });
    },
    onError: (error: any) => {
      console.error("User update error:", error);
      toast({
        title: "Update Failed",
        description: error.message || "Unable to update user. Please try again.",
        variant: "destructive"
      });
    }
  });

  // Delete user mutation
  const deleteUserMutation = useMutation({
    mutationFn: async (userId: number) => {
      const response = await apiRequest(`/api/system-users/${userId}`, "DELETE", {});
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/system-users"] });
      toast({
        title: "Success",
        description: "User deleted successfully"
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to delete user",
        variant: "destructive"
      });
    }
  });

  const handleCreateUser = () => {
    if (!newUser.username || !newUser.email || !newUser.password) {
      toast({
        title: "Error",
        description: "Please fill in all required fields",
        variant: "destructive"
      });
      return;
    }
    createUserMutation.mutate(newUser);
  };

  const handleEditUser = (systemUser: SystemUser) => {
    setEditingUser(systemUser);
    setEditUserData({
      username: systemUser.username,
      email: systemUser.email,
      password: "",
      role: systemUser.role,
      isActive: systemUser.isActive,
      pointsToAdd: ""
    });
    setIsEditModalOpen(true);
  };

  const handleUpdateUser = () => {
    if (!editingUser || !editUserData.username || !editUserData.email) {
      toast({
        title: "Error",
        description: "Please fill in all required fields",
        variant: "destructive"
      });
      return;
    }
    updateUserMutation.mutate({ userId: editingUser.id, userData: editUserData });
  };

  const handleDeleteUser = (userId: number, username: string) => {
    if (confirm(`Are you sure you want to delete user "${username}"? This action cannot be undone.`)) {
      deleteUserMutation.mutate(userId);
    }
  };

  const getRoleBadgeColor = (role: string) => {
    return role === "admin" 
      ? "bg-red-100 text-red-800" 
      : "bg-blue-100 text-blue-800";
  };

  const isCreating = createUserMutation.isPending;
  const isEditing = updateUserMutation.isPending;

  // Redirect if not admin
  if (user && user.role !== "admin") {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-gray-900 mb-2">Access Denied</h1>
          <p className="text-gray-600 mb-4">You need administrator privileges to access this page.</p>
          <Link href="/dashboard">
            <Button>Return to Dashboard</Button>
          </Link>
        </div>
      </div>
    );
  }

  if (isLoading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="w-8 h-8 border-4 border-primary border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
          <p className="text-muted-foreground">Loading users...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 flex flex-col lg:flex-row">
      {/* Mobile Header */}
      <div className="lg:hidden bg-white border-b border-gray-200 p-3">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-2">
            <svg width="20" height="20" viewBox="0 0 24 24" className="text-google-blue">
              <path fill="currentColor" d="M21.58 7.19c-.23-.86-.91-1.54-1.77-1.77C18.25 5 12 5 12 5s-6.25 0-7.81.42c-.86.23-1.54.91-1.77 1.77C2 8.75 2 12 2 12s0 3.25.42 4.81c.23.86.91 1.54 1.77 1.77C5.75 19 12 19 12 19s6.25 0 7.81-.42c.86-.23 1.54-.91 1.77-1.77C22 15.25 22 12 22 12s0-3.25-.42-4.81zM10 15V9l5.2 3L10 15z"/>
            </svg>
            <h1 className="text-lg font-medium text-gray-900">YouTube Manager</h1>
          </div>
          <div className="flex items-center space-x-2">
            <div className="text-right">
              <p className="text-xs font-medium text-gray-900">{user?.email?.split('@')[0] || 'Admin'}</p>
              <p className="text-xs text-gray-500">Admin</p>
            </div>
            <Button
              variant="outline"
              size="sm"
              onClick={logout}
              className="flex items-center space-x-1 px-2 py-1"
            >
              <LogOut className="w-3 h-3" />
            </Button>
          </div>
        </div>
      </div>

      {/* Mobile Navigation */}
      <nav className="lg:hidden bg-white border-b border-gray-200 px-3 py-2">
        <div className="flex space-x-1 overflow-x-auto">
          <Link href="/dashboard">
            <div className={`flex items-center space-x-2 px-3 py-2 rounded-md text-xs font-medium cursor-pointer whitespace-nowrap google-transition ${
              location === "/dashboard" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
            }`}>
              <span className="material-icons text-sm">dashboard</span>
              <span>Dashboard</span>
            </div>
          </Link>
          <Link href="/video-library">
            <div className={`flex items-center space-x-2 px-3 py-2 rounded-md text-xs font-medium cursor-pointer whitespace-nowrap google-transition ${
              location === "/video-library" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
            }`}>
              <span className="material-icons text-sm">video_library</span>
              <span>Videos</span>
            </div>
          </Link>
          <Link href="/analytics">
            <div className={`flex items-center space-x-2 px-3 py-2 rounded-md text-xs font-medium cursor-pointer whitespace-nowrap google-transition ${
              location === "/analytics" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
            }`}>
              <span className="material-icons text-sm">analytics</span>
              <span>Analytics</span>
            </div>
          </Link>
          <Link href="/orders">
            <div className={`flex items-center space-x-2 px-3 py-2 rounded-md text-xs font-medium cursor-pointer whitespace-nowrap google-transition ${
              location === "/orders" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
            }`}>
              <span className="material-icons text-sm">shopping_cart</span>
              <span>Orders</span>
            </div>
          </Link>
          <Link href="/system-users">
            <div className={`flex items-center space-x-2 px-3 py-2 rounded-md text-xs font-medium cursor-pointer whitespace-nowrap google-transition ${
              location === "/system-users" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
            }`}>
              <span className="material-icons text-sm">people</span>
              <span>Users</span>
            </div>
          </Link>
        </div>
      </nav>

      {/* Desktop Sidebar */}
      <div className="hidden lg:flex w-64 bg-white shadow-sm border-r border-gray-200 flex-col">
        {/* Header */}
        <div className="p-4 border-b border-gray-200">
          <div className="flex items-center space-x-3">
            <svg width="24" height="24" viewBox="0 0 24 24" className="text-google-blue">
              <path fill="currentColor" d="M21.58 7.19c-.23-.86-.91-1.54-1.77-1.77C18.25 5 12 5 12 5s-6.25 0-7.81.42c-.86.23-1.54.91-1.77 1.77C2 8.75 2 12 2 12s0 3.25.42 4.81c.23.86.91 1.54 1.77 1.77C5.75 19 12 19 12 19s6.25 0 7.81-.42c.86-.23 1.54-.91 1.77-1.77C22 15.25 22 12 22 12s0-3.25-.42-4.81zM10 15V9l5.2 3L10 15z"/>
            </svg>
            <div>
              <div className="flex items-center space-x-3">
                <h1 className="text-lg font-medium text-gray-900">YouTube Manager</h1>
                <div className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-semibold bg-gradient-to-r from-google-blue to-google-green text-white shadow-sm">
                  v3.2.7
                </div>
              </div>
              <p className="text-xs text-gray-500">Google Internal Tools</p>
            </div>
          </div>
        </div>

        {/* Navigation */}
        <nav className="flex-1 overflow-y-auto">
          <div className="p-4">
            <div className="space-y-1">
              <Link href="/dashboard">
                <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                  location === "/dashboard" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                }`}>
                  <span className="material-icons text-lg">dashboard</span>
                  <span>Dashboard</span>
                </div>
              </Link>
              <Link href="/video-library">
                <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                  location === "/video-library" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                }`}>
                  <span className="material-icons text-lg">video_library</span>
                  <span>Video Library</span>
                </div>
              </Link>
              <Link href="/analytics">
                <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                  location === "/analytics" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                }`}>
                  <span className="material-icons text-lg">analytics</span>
                  <span>Analytics</span>
                </div>
              </Link>
              <Link href="/api">
                <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                  location === "/api" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                }`}>
                  <span className="material-icons text-lg">api</span>
                  <span>API Services</span>
                </div>
              </Link>
              {user?.role === "admin" && (
                <Link href="/system-users">
                  <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                    location === "/system-users" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                  }`}>
                    <span className="material-icons text-lg">people</span>
                    <span>System Users</span>
                  </div>
                </Link>
              )}
              <div className="flex items-center space-x-3 px-3 py-2 rounded-md text-gray-600 hover:bg-gray-100 hover:text-gray-900 text-sm cursor-pointer google-transition">
                <span className="material-icons text-lg">tune</span>
                <span>Algorithm Tools</span>
              </div>
              <div className="flex items-center space-x-3 px-3 py-2 rounded-md text-gray-600 hover:bg-gray-100 hover:text-gray-900 text-sm cursor-pointer google-transition">
                <span className="material-icons text-lg">settings</span>
                <span>Settings</span>
              </div>
            </div>
          </div>
        </nav>

        {/* User Profile */}
        <div className="border-t border-gray-200 p-4">
          <div className="flex items-center space-x-3 mb-3">
            <div className="w-8 h-8 bg-gray-100 rounded-full flex items-center justify-center">
              <User className="w-4 h-4 text-gray-600" />
            </div>
            <div className="flex-1">
              <div className="text-sm font-medium text-gray-900">{user?.username}</div>
              <div className="text-xs text-gray-500">{user?.role === "admin" ? "Administrator" : "Client"}</div>
            </div>
          </div>
          <button
            onClick={logout}
            className="w-full flex items-center space-x-2 px-3 py-2 text-sm text-gray-600 hover:bg-gray-100 hover:text-gray-900 rounded-md google-transition"
          >
            <LogOut className="w-4 h-4" />
            <span>Logout</span>
          </button>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 flex flex-col">
        {/* Header */}
        <header className="hidden lg:block bg-white border-b border-gray-200 shadow-sm">
          <div className="max-w-full mx-auto px-6 py-3">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-4">
                <div className="flex items-center space-x-3">
                  <Users className="w-6 h-6 text-google-blue" />
                  <div>
                    <h1 className="text-xl font-medium text-gray-900">System Users</h1>
                    <p className="text-sm text-gray-500">User Management • Admin Panel</p>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </header>

        {/* Main Content */}
        <main className="flex-1 p-6 overflow-auto">
          <div className="flex items-center justify-between mb-6">
            <div className="flex items-center space-x-3">
              <Users className="w-6 h-6 text-google-blue" />
              <h2 className="text-2xl font-bold text-gray-900">System Users</h2>
              <Badge variant="secondary">{users?.length || 0} total</Badge>
            </div>
            
            <Dialog open={isCreateModalOpen} onOpenChange={setIsCreateModalOpen}>
              <DialogTrigger asChild>
                <Button className="bg-google-blue hover:bg-blue-600 text-white">
                  <UserPlus className="w-4 h-4 mr-2" />
                  Create User
                </Button>
              </DialogTrigger>
              <DialogContent className="sm:max-w-md">
                <DialogHeader>
                  <DialogTitle>Create New User</DialogTitle>
                  <DialogDescription>
                    Create a new user account with admin or client privileges.
                  </DialogDescription>
                </DialogHeader>
                <div className="space-y-4">
                  <div>
                    <Label htmlFor="username">Username</Label>
                    <Input
                      id="username"
                      value={newUser.username}
                      onChange={(e) => setNewUser({ ...newUser, username: e.target.value })}
                      placeholder="Enter username"
                    />
                  </div>
                  <div>
                    <Label htmlFor="email">Email</Label>
                    <Input
                      id="email"
                      type="email"
                      value={newUser.email}
                      onChange={(e) => setNewUser({ ...newUser, email: e.target.value })}
                      placeholder="Enter email address"
                    />
                  </div>
                  <div>
                    <Label htmlFor="password">Password</Label>
                    <Input
                      id="password"
                      type="password"
                      value={newUser.password}
                      onChange={(e) => setNewUser({ ...newUser, password: e.target.value })}
                      placeholder="Enter password"
                    />
                  </div>
                  <div>
                    <Label htmlFor="role">Role</Label>
                    <Select value={newUser.role} onValueChange={(value) => setNewUser({ ...newUser, role: value })}>
                      <SelectTrigger>
                        <SelectValue placeholder="Select role" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="client">Client</SelectItem>
                        <SelectItem value="admin">Admin</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="flex space-x-2 pt-4">
                    <Button
                      onClick={handleCreateUser}
                      disabled={isCreating}
                      className="flex-1 bg-google-blue hover:bg-blue-600"
                    >
                      {isCreating ? "Creating..." : "Create User"}
                    </Button>
                    <Button
                      variant="outline"
                      onClick={() => setIsCreateModalOpen(false)}
                      className="flex-1"
                    >
                      Cancel
                    </Button>
                  </div>
                </div>
              </DialogContent>
            </Dialog>

            {/* Edit User Modal */}
            <Dialog open={isEditModalOpen} onOpenChange={setIsEditModalOpen}>
              <DialogContent className="sm:max-w-md">
                <DialogHeader>
                  <DialogTitle>Edit User: {editingUser?.username}</DialogTitle>
                  <DialogDescription>
                    Update user information, change password, adjust role, or add points.
                  </DialogDescription>
                </DialogHeader>
                <div className="space-y-4">
                  <div>
                    <Label htmlFor="edit-username">Username</Label>
                    <Input
                      id="edit-username"
                      value={editUserData.username}
                      onChange={(e) => setEditUserData({ ...editUserData, username: e.target.value })}
                      placeholder="Enter username"
                    />
                  </div>
                  <div>
                    <Label htmlFor="edit-email">Email</Label>
                    <Input
                      id="edit-email"
                      type="email"
                      value={editUserData.email}
                      onChange={(e) => setEditUserData({ ...editUserData, email: e.target.value })}
                      placeholder="Enter email address"
                    />
                  </div>
                  <div>
                    <Label htmlFor="edit-password">New Password (leave empty to keep current)</Label>
                    <Input
                      id="edit-password"
                      type="password"
                      value={editUserData.password}
                      onChange={(e) => setEditUserData({ ...editUserData, password: e.target.value })}
                      placeholder="Enter new password (optional)"
                    />
                  </div>
                  <div>
                    <Label htmlFor="edit-role">Role</Label>
                    <Select 
                      value={editUserData.role} 
                      onValueChange={(value) => setEditUserData({ ...editUserData, role: value })}
                    >
                      <SelectTrigger>
                        <SelectValue placeholder="Select role" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="client">Client</SelectItem>
                        <SelectItem value="admin">Admin</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="flex items-center space-x-2">
                    <input
                      type="checkbox"
                      id="edit-active"
                      checked={editUserData.isActive}
                      onChange={(e) => setEditUserData({ ...editUserData, isActive: e.target.checked })}
                      className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500"
                    />
                    <Label htmlFor="edit-active">Active User</Label>
                  </div>
                  <div>
                    <Label htmlFor="edit-points">Add Points (optional)</Label>
                    <Input
                      id="edit-points"
                      type="number"
                      min="0"
                      max="1000"
                      value={editUserData.pointsToAdd}
                      onChange={(e) => setEditUserData({ ...editUserData, pointsToAdd: e.target.value })}
                      placeholder="Enter points to add (0-1000)"
                    />
                    <p className="text-xs text-gray-500 mt-1">
                      Add points to this user's current daily allocation. Will be logged in transaction history.
                    </p>
                  </div>
                  <div className="flex space-x-2 pt-4">
                    <Button
                      onClick={handleUpdateUser}
                      disabled={isEditing}
                      className="flex-1 bg-google-blue hover:bg-blue-600"
                    >
                      {isEditing ? "Updating..." : "Update User"}
                    </Button>
                    <Button
                      variant="outline"
                      onClick={() => setIsEditModalOpen(false)}
                      className="flex-1"
                    >
                      Cancel
                    </Button>
                  </div>
                </div>
              </DialogContent>
            </Dialog>
          </div>

          {/* Users Grid */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {users?.map((systemUser: SystemUser) => (
              <Card key={systemUser.id} className="hover:shadow-lg transition-shadow">
                <CardHeader className="space-y-0 pb-3">
                  <div className="flex items-start justify-between mb-3">
                    <div className="flex items-center space-x-3">
                      <div className={`w-10 h-10 rounded-full flex items-center justify-center ${
                        systemUser.role === "admin" ? "bg-red-50" : "bg-blue-50"
                      }`}>
                        {systemUser.role === "admin" ? (
                          <Shield className={`w-5 h-5 ${systemUser.role === "admin" ? "text-red-600" : "text-blue-600"}`} />
                        ) : (
                          <User className="w-5 h-5 text-blue-600" />
                        )}
                      </div>
                      <div>
                        <CardTitle className="text-lg">{systemUser.username}</CardTitle>
                        <p className="text-sm text-gray-500">{systemUser.email}</p>
                      </div>
                    </div>
                    <div className="flex items-center space-x-2">
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => handleEditUser(systemUser)}
                        className="text-blue-600 hover:text-blue-700 hover:bg-blue-50 p-2"
                        title="Edit User"
                      >
                        <Edit className="w-4 h-4" />
                      </Button>
                      {systemUser.id !== user?.id && (
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => handleDeleteUser(systemUser.id, systemUser.username)}
                          className="text-red-600 hover:text-red-700 hover:bg-red-50 p-2"
                          title="Delete User"
                        >
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      )}
                    </div>
                  </div>
                  {userPointsMap.get(systemUser.id) && (
                    <div className="bg-blue-50 rounded-lg p-3 mb-3">
                      <div className="flex items-center justify-between">
                        <span className="text-sm font-medium text-blue-900">Points Balance</span>
                        <div className="text-right">
                          <div className="text-lg font-bold text-blue-600">
                            {userPointsMap.get(systemUser.id)?.remainingPoints || 0}
                          </div>
                          <div className="text-xs text-blue-500">
                            of {userPointsMap.get(systemUser.id)?.totalPoints || 0} total ({userPointsMap.get(systemUser.id)?.usedPoints || 0} used)
                          </div>
                        </div>
                      </div>
                    </div>
                  )}
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-gray-500">Role:</span>
                      <Badge className={getRoleBadgeColor(systemUser.role)}>
                        {systemUser.role}
                      </Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-gray-500">Status:</span>
                      <Badge className={systemUser.isActive ? "bg-green-100 text-green-800" : "bg-gray-100 text-gray-800"}>
                        {systemUser.isActive ? "Active" : "Inactive"}
                      </Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-gray-500">Last Login:</span>
                      <span className="text-sm text-gray-900">
                        {systemUser.lastLogin 
                          ? new Date(systemUser.lastLogin).toLocaleDateString()
                          : "Never"
                        }
                      </span>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm text-gray-500">Created:</span>
                      <span className="text-sm text-gray-900">
                        {new Date(systemUser.createdAt).toLocaleDateString()}
                      </span>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>

          {users?.length === 0 && (
            <div className="text-center py-12">
              <Users className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">No users found</h3>
              <p className="text-gray-500 mb-6">Get started by creating your first user.</p>
              <Button onClick={() => setIsCreateModalOpen(true)} className="bg-google-blue hover:bg-blue-600">
                <UserPlus className="w-4 h-4 mr-2" />
                Create First User
              </Button>
            </div>
          )}
        </main>
      </div>
    </div>
  );
}