import OpenAI from "openai";

// the newest OpenAI model is "gpt-4o" which was released May 13, 2024. do not change this unless explicitly requested by the user
const openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });

export interface WhatsAppGraphic {
  url: string;
  description: string;
}

export class OpenAIService {
  static async generateYouTubeAdvice(videoData: {
    title: string;
    description: string | null;
    views: number;
    duration: string | null;
    publishedAt: string | null;
    algorithmScore: number;
    ctr: number;
    engagementRate: number;
    seoScore: number;
  }): Promise<string> {
    try {
      const prompt = `As a YouTube optimization expert, analyze this video's metadata and provide specific advice to boost its ranking and visibility WITHOUT changing the title. 

Video Details:
- Title: "${videoData.title}"
- Description: "${videoData.description || 'No description'}"
- Views: ${videoData.views.toLocaleString()}
- Duration: ${videoData.duration || 'Unknown'}
- Published: ${videoData.publishedAt || 'Unknown'}
- Algorithm Score: ${videoData.algorithmScore}/10
- Click-Through Rate: ${videoData.ctr}%
- Engagement Rate: ${videoData.engagementRate}%
- SEO Score: ${videoData.seoScore}/100

Provide actionable advice in exactly this JSON format:
{
  "keyStrengths": "Brief summary of video's current strengths",
  "mainIssues": "Primary areas needing improvement",
  "actionableSteps": [
    "Specific step 1 to improve visibility",
    "Specific step 2 to boost engagement", 
    "Specific step 3 to enhance discoverability"
  ],
  "expectedImpact": "Realistic outcome expectations",
  "timeframe": "Implementation timeline"
}`;

      const response = await openai.chat.completions.create({
        model: "gpt-4o",
        messages: [
          {
            role: "system",
            content: "You are a YouTube algorithm optimization expert. Provide specific, actionable advice based on video metadata. Always respond in valid JSON format."
          },
          {
            role: "user",
            content: prompt
          }
        ],
        response_format: { type: "json_object" },
        max_tokens: 800
      });

      return response.choices[0].message.content || "Unable to generate advice at this time.";
    } catch (error) {
      console.error("Failed to generate YouTube advice:", error);
      return JSON.stringify({
        keyStrengths: "Video analysis pending",
        mainIssues: "Optimization recommendations unavailable",
        actionableSteps: ["Contact support for personalized advice"],
        expectedImpact: "Analysis required",
        timeframe: "N/A"
      });
    }
  }

  static async generateWhatsAppBoostGraphic(channelName: string, step: string): Promise<WhatsAppGraphic> {
    try {
      const prompt = `Create a high-tech, futuristic dashboard interface graphic for "${step}" showing YouTube exposure boost analytics for channel "${channelName}". 
      Style: Professional Google Material Design with YouTube red (#FF0000) accents, dark theme with neon highlights, 
      holographic UI elements, real-time data visualization charts, network nodes connecting globally, 
      viral spread animations, engagement metrics with glowing indicators, YouTube integration panels, 
      YouTube-style thumbnails floating in 3D space, subscriber count displays, view analytics, 
      professional corporate dashboard aesthetic. Ultra-modern, sci-fi inspired, high-tech visualization 
      that looks like advanced YouTube Creator Studio internal tools with exposure boost indicators.`;

      const response = await openai.images.generate({
        model: "dall-e-3",
        prompt: prompt,
        n: 1,
        size: "1024x1024",
        quality: "hd",
      });

      return {
        url: response.data?.[0]?.url || "",
        description: step
      };
    } catch (error) {
      console.error("Failed to generate YouTube graphic:", error);
      // Return a fallback description without an image
      return {
        url: "",
        description: step
      };
    }
  }

  static async generateMultipleWhatsAppGraphics(channelName: string, steps: string[]): Promise<WhatsAppGraphic[]> {
    const graphics: WhatsAppGraphic[] = [];
    
    // Generate graphics for key steps (not all to avoid rate limits)
    const keySteps = [steps[0], steps[Math.floor(steps.length/2)], steps[steps.length-1]];
    
    for (const step of keySteps) {
      const graphic = await this.generateWhatsAppBoostGraphic(channelName, step);
      graphics.push(graphic);
      
      // Add delay to respect rate limits
      await new Promise(resolve => setTimeout(resolve, 1000));
    }
    
    return graphics;
  }
}