import { storage } from "./storage";

// Simple daily point reset scheduler
export class PointScheduler {
  private static resetInterval: NodeJS.Timeout | null = null;

  static start() {
    // Check for daily reset every hour
    this.resetInterval = setInterval(async () => {
      try {
        await this.checkAndResetDailyPoints();
      } catch (error) {
        console.error('Error in daily point reset check:', error);
      }
    }, 60 * 60 * 1000); // Check every hour

    console.log('Point scheduler started - checking for daily resets every hour');
  }

  static stop() {
    if (this.resetInterval) {
      clearInterval(this.resetInterval);
      this.resetInterval = null;
      console.log('Point scheduler stopped');
    }
  }

  private static async checkAndResetDailyPoints() {
    const today = new Date().toISOString().split('T')[0];
    const users = await storage.getAllUsers();
    const defaultPointsSetting = await storage.getSystemSetting('default_daily_points');
    const defaultPoints = defaultPointsSetting ? parseInt(defaultPointsSetting.value) : 600;

    let resetCount = 0;

    for (const user of users) {
      const existingAllocation = await storage.getUserPointAllocation(user.id, today);
      
      if (!existingAllocation) {
        // Create new allocation for today
        await storage.createUserPointAllocation({
          userId: user.id,
          date: today,
          totalPoints: defaultPoints,
          usedPoints: 0,
          remainingPoints: defaultPoints,
          isCustomAllocation: false
        });
        resetCount++;
      }
    }

    if (resetCount > 0) {
      console.log(`Daily point reset: Created allocations for ${resetCount} users with ${defaultPoints} points each`);
    }

    return resetCount;
  }

  // Manual reset function for admin use
  static async manualReset(): Promise<{ resetCount: number; totalUsers: number; defaultPoints: number }> {
    const today = new Date().toISOString().split('T')[0];
    const users = await storage.getAllUsers();
    const defaultPointsSetting = await storage.getSystemSetting('default_daily_points');
    const defaultPoints = defaultPointsSetting ? parseInt(defaultPointsSetting.value) : 600;

    let resetCount = 0;

    for (const user of users) {
      const existingAllocation = await storage.getUserPointAllocation(user.id, today);
      
      if (!existingAllocation) {
        await storage.createUserPointAllocation({
          userId: user.id,
          date: today,
          totalPoints: defaultPoints,
          usedPoints: 0,
          remainingPoints: defaultPoints,
          isCustomAllocation: false
        });
        resetCount++;
      }
    }

    return {
      resetCount,
      totalUsers: users.length,
      defaultPoints
    };
  }
}