import { storage } from "./storage";
import type { QueueItem, InsertQueueItem } from "@shared/schema";
import { generateExposeStepContent } from "./anthropic";

export class QueueManager {
  private static processingInterval: NodeJS.Timeout | null = null;
  private static stuckCheckInterval: NodeJS.Timeout | null = null;
  private static cleanupInterval: NodeJS.Timeout | null = null;
  private static readonly PROCESS_INTERVAL = 3000; // Check every 3 seconds for faster processing
  private static readonly STUCK_CHECK_INTERVAL = 30000; // Check for stuck items every 30 seconds
  private static readonly CLEANUP_INTERVAL = 60 * 60 * 1000; // Clean up every hour
  private static readonly STUCK_TIMEOUT = 5 * 60 * 1000; // 5 minutes timeout for stuck detection
  private static readonly SIMULATION_CLEANUP_TIMEOUT = 12 * 60 * 60 * 1000; // 12 hours for simulation cleanup
  private static readonly MAX_RETRY_ATTEMPTS = 3;
  
  // Define which action types are simulations vs actual service orders
  private static readonly SIMULATION_ACTIONS = ['boost', 'session', 'replenish', 'devalue', 'whatsapp'];
  private static readonly SERVICE_ACTIONS = ['video_views', 'subscribers', 'likes']; // Preserve these permanently
  
  static start() {
    if (this.processingInterval) {
      return; // Already running
    }
    
    console.log("Queue manager started with anti-stuck mechanisms and auto-cleanup");
    this.processingInterval = setInterval(() => {
      this.processQueue().catch(console.error);
    }, this.PROCESS_INTERVAL);
    
    // Start stuck item checker
    this.stuckCheckInterval = setInterval(() => {
      this.checkAndRecoverStuckItems().catch(console.error);
    }, this.STUCK_CHECK_INTERVAL);
    
    // Start simulation cleanup checker
    this.cleanupInterval = setInterval(() => {
      this.cleanupOldSimulations().catch(console.error);
    }, this.CLEANUP_INTERVAL);
    
    // Run initial cleanup
    this.cleanupOldSimulations().catch(console.error);
  }
  
  static stop() {
    if (this.processingInterval) {
      clearInterval(this.processingInterval);
      this.processingInterval = null;
    }
    if (this.stuckCheckInterval) {
      clearInterval(this.stuckCheckInterval);
      this.stuckCheckInterval = null;
    }
    if (this.cleanupInterval) {
      clearInterval(this.cleanupInterval);
      this.cleanupInterval = null;
    }
    console.log("Queue manager stopped");
  }

  /**
   * Clean up old completed simulation queue items (12+ hours old)
   * Preserves actual service orders (views, subscribers, likes) permanently
   */
  private static async cleanupOldSimulations() {
    try {
      console.log("🧹 Cleaning up old simulation queue items...");
      
      const cutoffTime = new Date(Date.now() - this.SIMULATION_CLEANUP_TIMEOUT);
      
      // Get completed simulation items older than 12 hours
      const oldSimulations = await storage.getCompletedSimulationItems(cutoffTime, this.SIMULATION_ACTIONS);
      
      if (oldSimulations.length > 0) {
        console.log(`🗑️ Found ${oldSimulations.length} old simulation items to clean up`);
        
        // Delete old simulation items
        await storage.deleteOldSimulationItems(cutoffTime, this.SIMULATION_ACTIONS);
        
        console.log(`✅ Cleaned up ${oldSimulations.length} old simulation queue items`);
      }
    } catch (error) {
      console.error("❌ Error during simulation cleanup:", error);
    }
  }

  /**
   * Check for stuck items and automatically recover them
   */
  private static async checkAndRecoverStuckItems() {
    try {
      console.log("🔍 Checking for stuck queue items...");
      
      // Find items that have been running for too long
      const stuckRunningItems = await storage.getStuckRunningItems(this.STUCK_TIMEOUT);
      
      // Find items that have been queued for too long without processing
      const stuckQueuedItems = await storage.getStuckQueuedItems(this.STUCK_TIMEOUT);
      
      const allStuckItems = [...stuckRunningItems, ...stuckQueuedItems];
      
      if (allStuckItems.length > 0) {
        console.log(`⚠️  Found ${allStuckItems.length} stuck queue items. Initiating recovery...`);
        
        for (const item of allStuckItems) {
          await this.recoverStuckItem(item);
        }
      }
    } catch (error) {
      console.error("Error checking for stuck items:", error);
    }
  }

  /**
   * Recover a single stuck item
   */
  private static async recoverStuckItem(item: QueueItem) {
    try {
      // Force complete stuck items instead of retrying to prevent infinite loops
      console.log(`🔄 Force completing stuck queue item ${item.id} - ${item.actionType} for channel ${item.channelId}`);
      
      await storage.updateQueueItem(item.id, {
        status: "completed",
        progress: 100,
        currentStep: "Auto-completed - recovered from stuck state",
        completedAt: new Date()
      });
      
    } catch (error) {
      console.error(`Error recovering stuck item ${item.id}:`, error);
    }
  }

  /**
   * Force complete stuck items (admin function)
   */
  static async forceCompleteStuckItems(userId?: number): Promise<number> {
    try {
      const stuckItems = userId 
        ? await storage.getStuckItemsByUser(userId)
        : await storage.getAllStuckItems();
      
      console.log(`🚨 Force completing ${stuckItems.length} stuck items${userId ? ` for user ${userId}` : ''}...`);
      
      let completedCount = 0;
      for (const item of stuckItems) {
        await storage.updateQueueItem(item.id, {
          status: "completed",
          progress: 100,
          currentStep: "Force completed by admin",
          completedAt: new Date()
        });
        completedCount++;
      }
      
      console.log(`✅ Force completed ${completedCount} stuck queue items`);
      return completedCount;
    } catch (error) {
      console.error("Error force completing stuck items:", error);
      return 0;
    }
  }
  
  private static async processQueue() {
    try {
      // Get all users with queued items (this is a simplified approach)
      // In a real application, you might want to track active users differently
      const allUsers = await storage.getAllUsers();
      
      for (const user of allUsers) {
        await this.processUserQueue(user.id);
      }
    } catch (error) {
      console.error("Error processing queue:", error);
    }
  }
  
  private static async processUserQueue(userId: number) {
    try {
      // Check if user has any running simulations
      const runningItems = await storage.getRunningQueueItems(userId);
      
      // Limit concurrent simulations per user (e.g., max 2)
      const maxConcurrent = 2;
      if (runningItems.length >= maxConcurrent) {
        return; // User already has max concurrent simulations
      }
      
      // Get next queued item
      const nextItem = await storage.getNextQueuedItem(userId);
      if (!nextItem) {
        return; // No queued items
      }
      
      // Start the simulation
      await this.startQueueItem(nextItem);
    } catch (error) {
      console.error(`Error processing queue for user ${userId}:`, error);
    }
  }
  
  private static async startQueueItem(item: QueueItem) {
    try {
      console.log(`Starting queue item ${item.id}: ${item.actionType} for channel ${item.channelId}`);
      
      // Update status to running
      await storage.updateQueueItem(item.id, {
        status: "running",
        startedAt: new Date(),
        progress: 0,
        currentStep: "Initializing..."
      });
      
      // Start the actual simulation based on action type
      this.executeSimulation(item).catch(async (error) => {
        console.error(`Simulation failed for queue item ${item.id}:`, error);
        await storage.updateQueueItem(item.id, {
          status: "failed",
          currentStep: error.message || "Simulation failed",
          completedAt: new Date()
        });
      });
      
    } catch (error) {
      console.error(`Error starting queue item ${item.id}:`, error);
    }
  }
  
  private static async executeSimulation(item: QueueItem) {
    const steps = this.getSimulationSteps(item.actionType);
    
    try {
      for (let i = 0; i < steps.length; i++) {
        const progress = Math.floor(((i + 1) / steps.length) * 100);
        
        // Generate dynamic AI content for expose simulation steps
        let currentStepText = steps[i];
        if (item.actionType === 'whatsapp') {
          try {
            const aiContent = await generateExposeStepContent(i + 1, steps.length, steps[i]);
            if (aiContent) {
              currentStepText = aiContent;
            }
          } catch (error) {
            console.error(`Failed to generate AI content for step ${i + 1}:`, error);
            // Fall back to original step text
          }
        }
        
        // Update progress with heartbeat timestamp to prevent stuck detection
        await storage.updateQueueItem(item.id, {
          progress,
          currentStep: currentStepText,
          status: 'running',
          updatedAt: new Date() // Critical: Always update timestamp for anti-stuck detection
        });
        
        console.log(`Queue item ${item.id} - Step ${i + 1}/${steps.length}: ${currentStepText} (${progress}%)`);
        
        // Check if item was cancelled or marked for completion
        const currentItem = await storage.getQueueItemById(item.id);
        if (!currentItem || currentItem.status === "cancelled" || currentItem.status === "completed") {
          console.log(`Queue item ${item.id} was ${currentItem?.status || 'cancelled'}`);
          return;
        }
        
        // Dynamic step duration based on action type - shorter for expose simulation
        const stepDuration = item.actionType === 'whatsapp' ? 8000 : 6000; // 8 seconds for expose, 6 for others
        if (i < steps.length - 1) {
          // Send heartbeat every 3 seconds during wait to prevent stuck detection
          const heartbeatInterval = setInterval(async () => {
            try {
              await storage.updateQueueItem(item.id, {
                updatedAt: new Date() // Heartbeat to show item is actively processing
              });
            } catch (error) {
              console.error(`Heartbeat failed for item ${item.id}:`, error);
            }
          }, 3000);
          
          await new Promise(resolve => setTimeout(resolve, stepDuration));
          clearInterval(heartbeatInterval);
        }
      }
      
      // Mark as completed with final timestamp
      await storage.updateQueueItem(item.id, {
        status: "completed",
        progress: 100,
        currentStep: "Completed successfully",
        completedAt: new Date(),
        updatedAt: new Date()
      });
      
      console.log(`✅ Queue item ${item.id} completed: ${item.actionType} for channel ${item.channelId}`);
    } catch (error) {
      console.error(`❌ Error executing queue item ${item.id}:`, error);
      
      // Mark as failed with timestamp
      await storage.updateQueueItem(item.id, {
        status: "failed",
        currentStep: "Failed - Please try again",
        completedAt: new Date(),
        updatedAt: new Date()
      });
    }
  }
  
  private static getSimulationSteps(actionType: string): string[] {
    switch (actionType) {
      case "boost":
        return [
          "Analyzing video metadata and tags...",
          "Optimizing thumbnail positioning...",
          "Enhancing title for SEO performance...",
          "Adjusting description keywords...",
          "Calibrating engagement metrics...",
          "Updating algorithm weights...",
          "Finalizing optimization parameters...",
          "Synchronizing with YouTube servers..."
        ];
      case "session":
        return [
          "Establishing secure connection to YouTube API",
          "Analyzing current video performance metrics",
          "Identifying algorithm optimization opportunities", 
          "Implementing real-time engagement boosts",
          "Synchronizing with recommendation engine",
          "Applying advanced ranking factors",
          "Optimizing visual appeal algorithms",
          "Finalizing algorithm grade improvements"
        ];
      case "replenish":
        return [
          "Connecting to Google Algorithm servers...",
          "Authenticating channel credentials...",
          "Analyzing current algorithm performance...",
          "Calculating optimal point allocation...",
          "Initializing point transfer protocol...",
          "Synchronizing with YouTube backend...",
          "Applying algorithm grade improvements...",
          "Validating score consistency...",
          "Updating channel metrics...",
          "Finalizing AG point distribution..."
        ];
      case "devalue":
        return [
          "Initializing devaluation simulation...",
          "Analyzing channel algorithm parameters...",
          "Running negative impact scenarios...",
          "Calculating score reduction factors...",
          "Simulating competitor advantages...",
          "Processing algorithm penalties...",
          "Applying devaluation algorithms...",
          "Updating recommendation weights...",
          "Adjusting algorithm grade points...",
          "Finalizing devaluation process..."
        ];
      case "whatsapp":
        return [
          "🚀 Connecting to YouTube Discovery Engine...",
          "📊 Analyzing Global Trending Patterns & Algorithms...",
          "🌐 Deploying Multi-Platform YouTube Promotion Campaigns...",
          "💬 Activating 500+ YouTube Community Networks...",
          "🤖 Generating AI-Powered Thumbnails & Content Assets...",
          "🎯 Targeting High-Value YouTube Demographics & Creators...",
          "⚡ Amplifying Content Through YouTube Recommendation System...",
          "🔥 Optimizing Search Visibility & Discovery Algorithms...",
          "📱 Creating YouTube Shorts & Cross-Platform Assets...",
          "🌟 Activating Premium YouTube Exposure Protocol...",
          "✅ YouTube Channel Exposure Boost Successfully Deployed!"
        ];
      default:
        return ["Processing..."];
    }
  }
  
  static async addToQueue(queueItem: InsertQueueItem): Promise<QueueItem> {
    // Set estimated duration based on action type
    const estimatedDuration = this.getEstimatedDuration(queueItem.actionType);
    
    const itemWithDuration = {
      ...queueItem,
      estimatedDuration
    };
    
    return await storage.createQueueItem(itemWithDuration);
  }
  
  private static getEstimatedDuration(actionType: string): number {
    // Return duration in seconds
    switch (actionType) {
      case "boost":
        return 120; // 2 minutes
      case "session":
        return 90;  // 1.5 minutes
      case "replenish":
        return 150; // 2.5 minutes
      case "devalue":
        return 120; // 2 minutes
      case "whatsapp":
        return 90; // 1.5 minutes
      default:
        return 60;  // 1 minute default
    }
  }
  
  static async cancelQueueItem(itemId: number): Promise<void> {
    await storage.updateQueueItem(itemId, {
      status: "cancelled",
      completedAt: new Date()
    });
  }
  
  static async pauseQueueItem(itemId: number): Promise<void> {
    await storage.updateQueueItem(itemId, {
      status: "paused"
    });
  }
  
  static async resumeQueueItem(itemId: number): Promise<void> {
    await storage.updateQueueItem(itemId, {
      status: "queued" // Will be picked up by the next processing cycle
    });
  }
}