import crypto from 'crypto';
import { db } from './db';
import { deviceFingerprints, ipAddressLogs, securityViolations, users } from '@shared/schema';
import { eq, and, or, gte, sql } from 'drizzle-orm';

export interface DeviceFingerprintData {
  userAgent: string;
  screenResolution: string;
  timezone: string;
  language: string;
  platform: string;
  webglRenderer?: string;
  canvasFingerprint?: string;
  audioFingerprint?: string;
  macAddress?: string;
  networkInterfaces?: string;
  hardwareFingerprint?: string;
  browserFingerprint?: string;
}

export interface IpLocationData {
  country?: string;
  region?: string;
  city?: string;
  isp?: string;
  isVpn?: boolean;
  isProxy?: boolean;
}

export class SecurityService {
  private static readonly MAX_ACCOUNTS_PER_HOUSEHOLD = 1;
  private static readonly VPN_CHECK_ENABLED = true;
  private static readonly FINGERPRINT_SIMILARITY_THRESHOLD = 0.85;

  // Whitelist for development/testing devices - allows unlimited accounts
  private static readonly WHITELISTED_IPS = [
    '172.31.128.60', // Development/testing device
    '127.0.0.1',     // Localhost
    '::1'            // IPv6 localhost
  ];

  private static readonly WHITELISTED_USER_AGENTS = [
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/138.0.0.0 Safari/537.36'
  ];

  /**
   * Check if a device/IP is whitelisted for unlimited registrations
   */
  private static isDeviceWhitelisted(ipAddress: string, userAgent?: string): boolean {
    // Check IP whitelist
    if (this.WHITELISTED_IPS.includes(ipAddress)) {
      return true;
    }

    // Check user agent whitelist
    if (userAgent && this.WHITELISTED_USER_AGENTS.includes(userAgent)) {
      return true;
    }

    return false;
  }

  /**
   * Generate a unique device fingerprint hash from collected data
   */
  static generateFingerprint(data: DeviceFingerprintData): string {
    const fingerprintString = [
      data.userAgent,
      data.screenResolution,
      data.timezone,
      data.language,
      data.platform,
      data.webglRenderer || '',
      data.canvasFingerprint || '',
      data.audioFingerprint || '',
      data.hardwareFingerprint || '',
      data.browserFingerprint || ''
    ].join('|');

    return crypto.createHash('sha256').update(fingerprintString).digest('hex');
  }

  /**
   * Hash IP address for privacy while maintaining ability to detect duplicates
   */
  static hashIpAddress(ipAddress: string): string {
    return crypto.createHash('sha256').update(ipAddress).digest('hex');
  }

  /**
   * Check if registration should be blocked based on security rules
   */
  static async checkRegistrationSecurity(
    ipAddress: string,
    fingerprintData: DeviceFingerprintData,
    locationData?: IpLocationData,
    userAgent?: string
  ): Promise<{
    allowed: boolean;
    reason?: string;
    violations: string[];
  }> {
    // Check if device is whitelisted - if so, allow unlimited registrations
    if (this.isDeviceWhitelisted(ipAddress, userAgent)) {
      console.log(`✅ [SECURITY] Device whitelisted - bypassing security checks for IP: ${ipAddress}`);
      return { allowed: true, violations: [] };
    }

    const violations: string[] = [];
    const fingerprint = this.generateFingerprint(fingerprintData);
    const hashedIp = this.hashIpAddress(ipAddress);

    // Check for existing device fingerprint
    const existingFingerprint = await db
      .select()
      .from(deviceFingerprints)
      .where(eq(deviceFingerprints.fingerprintHash, fingerprint))
      .limit(1);

    if (existingFingerprint.length > 0) {
      violations.push('Device fingerprint already registered');
    }

    // Check IP address registration count
    const ipRecord = await db
      .select()
      .from(ipAddressLogs)
      .where(eq(ipAddressLogs.ipHash, hashedIp))
      .limit(1);

    if (ipRecord.length > 0 && ipRecord[0].userCount >= this.MAX_ACCOUNTS_PER_HOUSEHOLD) {
      violations.push(`Maximum accounts (${this.MAX_ACCOUNTS_PER_HOUSEHOLD}) already registered from this IP address`);
    }

    // Check for VPN/Proxy usage
    if (this.VPN_CHECK_ENABLED && locationData) {
      if (locationData.isVpn) {
        violations.push('VPN usage detected');
      }
      if (locationData.isProxy) {
        violations.push('Proxy usage detected');
      }
    }

    // Check for similar MAC addresses if available
    if (fingerprintData.macAddress) {
      const similarMacDevices = await db
        .select()
        .from(deviceFingerprints)
        .where(eq(deviceFingerprints.macAddress, fingerprintData.macAddress))
        .limit(1);

      if (similarMacDevices.length > 0) {
        violations.push('MAC address already registered');
      }
    }

    // Check for hardware fingerprint similarity
    if (fingerprintData.hardwareFingerprint) {
      const similarHardware = await db
        .select()
        .from(deviceFingerprints)
        .where(eq(deviceFingerprints.hardwareFingerprint, fingerprintData.hardwareFingerprint))
        .limit(1);

      if (similarHardware.length > 0) {
        violations.push('Similar hardware configuration already registered');
      }
    }

    const allowed = violations.length === 0;
    const reason = violations.length > 0 ? violations.join('; ') : undefined;

    return { allowed, reason, violations };
  }

  /**
   * Record a new device fingerprint for a user
   */
  static async recordDeviceFingerprint(
    userId: number,
    ipAddress: string,
    fingerprintData: DeviceFingerprintData,
    userAgent?: string
  ): Promise<number | null> {
    // Skip recording for whitelisted devices to avoid constraint violations
    if (this.isDeviceWhitelisted(ipAddress, userAgent)) {
      console.log(`🔐 [SECURITY] Skipping fingerprint recording for whitelisted device: ${ipAddress}`);
      return null;
    }

    const fingerprint = this.generateFingerprint(fingerprintData);

    try {
      const [fingerprintRecord] = await db
        .insert(deviceFingerprints)
        .values({
          userId: userId.toString(),
          fingerprintHash: fingerprint,
          ipAddress,
          userAgent: fingerprintData.userAgent,
          screenResolution: fingerprintData.screenResolution,
          timezone: fingerprintData.timezone,
          language: fingerprintData.language,
          platform: fingerprintData.platform,
          webglRenderer: fingerprintData.webglRenderer || null,
          canvasFingerprint: fingerprintData.canvasFingerprint || null,
          audioFingerprint: fingerprintData.audioFingerprint || null,
          macAddress: fingerprintData.macAddress || null,
          networkInterfaces: fingerprintData.networkInterfaces || null,
          hardwareFingerprint: fingerprintData.hardwareFingerprint || null,
          browserFingerprint: fingerprintData.browserFingerprint || null,
        })
        .returning({ id: deviceFingerprints.id });

      return fingerprintRecord.id;
    } catch (error) {
      // Handle duplicate fingerprint gracefully (e.g., from multiple accounts on same device)
      if (error.message && error.message.includes('duplicate key value violates unique constraint')) {
        console.log(`🔐 [SECURITY] Device fingerprint already exists, skipping duplicate recording`);
        return null;
      }
      throw error;
    }
  }

  /**
   * Update IP address tracking
   */
  static async updateIpAddressLog(
    ipAddress: string,
    locationData?: IpLocationData
  ): Promise<void> {
    const hashedIp = this.hashIpAddress(ipAddress);

    const existingRecord = await db
      .select()
      .from(ipAddressLogs)
      .where(eq(ipAddressLogs.ipHash, hashedIp))
      .limit(1);

    if (existingRecord.length > 0) {
      // Update existing record
      await db
        .update(ipAddressLogs)
        .set({
          userCount: sql`${ipAddressLogs.userCount} + 1`,
          lastSeen: new Date(),
          ...(locationData && {
            country: locationData.country,
            region: locationData.region,
            city: locationData.city,
            isp: locationData.isp,
            isVpn: locationData.isVpn,
            isProxy: locationData.isProxy,
          }),
        })
        .where(eq(ipAddressLogs.ipHash, hashedIp));
    } else {
      // Create new record
      await db
        .insert(ipAddressLogs)
        .values({
          ipAddress,
          ipHash: hashedIp,
          registrationCount: 1,
          ...(locationData && {
            country: locationData.country,
            region: locationData.region,
            city: locationData.city,
            isp: locationData.isp,
            isVpn: locationData.isVpn,
            isProxy: locationData.isProxy,
          }),
        });
    }
  }

  /**
   * Log a security violation
   */
  static async logSecurityViolation(
    violationType: string,
    description: string,
    severity: 'low' | 'medium' | 'high' | 'critical',
    actionTaken: string,
    userId?: number,
    fingerprintId?: number,
    ipAddress?: string,
    userAgent?: string,
    additionalData?: any
  ): Promise<void> {
    await db
      .insert(securityViolations)
      .values({
        userId,
        fingerprintId,
        violationType,
        description,
        severity,
        actionTaken,
        ipAddress,
        userAgent,
        additionalData: additionalData ? JSON.stringify(additionalData) : null,
      });
  }

  /**
   * Block a device fingerprint
   */
  static async blockDeviceFingerprint(
    fingerprintId: number,
    reason: string
  ): Promise<void> {
    await db
      .update(deviceFingerprints)
      .set({
        isBlocked: true,
        blockedReason: reason,
        updatedAt: new Date(),
      })
      .where(eq(deviceFingerprints.id, fingerprintId));
  }

  /**
   * Check if a device is blocked
   */
  static async isDeviceBlocked(fingerprint: string): Promise<boolean> {
    const [device] = await db
      .select({ isBlocked: deviceFingerprints.isBlocked })
      .from(deviceFingerprints)
      .where(
        and(
          eq(deviceFingerprints.fingerprintHash, fingerprint),
          eq(deviceFingerprints.isBlocked, true)
        )
      )
      .limit(1);

    return device ? device.isBlocked : false;
  }

  /**
   * Get security statistics for admin dashboard
   */
  static async getSecurityStats(): Promise<{
    totalFingerprints: number;
    blockedDevices: number;
    totalViolations: number;
    recentViolations: number;
    ipRegistrations: number;
    vpnDetections: number;
  }> {
    const thirtyDaysAgo = new Date(Date.now() - 30 * 24 * 60 * 60 * 1000);

    const [stats] = await db
      .select({
        totalFingerprints: sql<number>`count(distinct ${deviceFingerprints.id})`,
        blockedDevices: sql<number>`count(case when ${deviceFingerprints.isBlocked} then 1 end)`,
      })
      .from(deviceFingerprints);

    const [violationStats] = await db
      .select({
        totalViolations: sql<number>`count(*)`,
        recentViolations: sql<number>`count(case when ${securityViolations.createdAt} >= ${thirtyDaysAgo} then 1 end)`,
      })
      .from(securityViolations);

    const [ipStats] = await db
      .select({
        ipRegistrations: sql<number>`count(*)`,
        vpnDetections: sql<number>`count(case when ${ipAddressLogs.isVpn} = true then 1 end)`,
      })
      .from(ipAddressLogs);

    return {
      totalFingerprints: stats.totalFingerprints || 0,
      blockedDevices: stats.blockedDevices || 0,
      totalViolations: violationStats.totalViolations || 0,
      recentViolations: violationStats.recentViolations || 0,
      ipRegistrations: ipStats.ipRegistrations || 0,
      vpnDetections: ipStats.vpnDetections || 0,
    };
  }

  /**
   * Get recent security violations for admin review
   */
  static async getRecentViolations(limit = 50): Promise<any[]> {
    return await db
      .select({
        id: securityViolations.id,
        userId: securityViolations.userId,
        username: users.username,
        violationType: securityViolations.violationType,
        description: securityViolations.description,
        severity: securityViolations.severity,
        actionTaken: securityViolations.actionTaken,
        ipAddress: securityViolations.ipAddress,
        createdAt: securityViolations.createdAt,
      })
      .from(securityViolations)
      .leftJoin(users, eq(securityViolations.userId, users.id))
      .orderBy(sql`${securityViolations.createdAt} desc`)
      .limit(limit);
  }

  /**
   * Validate device fingerprint data completeness - now more lenient
   */
  static validateFingerprintData(data: any): DeviceFingerprintData | null {
    // Only require userAgent and platform as minimum requirements
    if (!data || !data.userAgent || !data.platform) {
      console.log('🔴 [SECURITY] Device fingerprint validation failed: missing required fields', {
        hasUserAgent: !!data?.userAgent,
        hasPlatform: !!data?.platform,
        receivedData: data
      });
      return null;
    }

    return {
      userAgent: String(data.userAgent),
      screenResolution: data.screenResolution ? String(data.screenResolution) : 'unknown',
      timezone: data.timezone ? String(data.timezone) : 'unknown',
      language: data.language ? String(data.language) : 'unknown',
      platform: String(data.platform),
      webglRenderer: data.webglRenderer ? String(data.webglRenderer) : undefined,
      canvasFingerprint: data.canvasFingerprint ? String(data.canvasFingerprint) : undefined,
      audioFingerprint: data.audioFingerprint ? String(data.audioFingerprint) : undefined,
      macAddress: data.macAddress ? String(data.macAddress) : undefined,
      networkInterfaces: data.networkInterfaces ? JSON.stringify(data.networkInterfaces) : undefined,
      hardwareFingerprint: data.hardwareFingerprint ? String(data.hardwareFingerprint) : undefined,
      browserFingerprint: data.browserFingerprint ? String(data.browserFingerprint) : undefined,
    };
  }
}