interface TubeBoostOrderStatus {
  charge: string;
  start_count: string;
  status: string;
  remains: string;
  currency: string;
}

interface TubeBoostStatusResponse {
  [orderId: string]: TubeBoostOrderStatus | { error: string };
}

export class TubeBoostAPI {
  private static readonly API_KEY = process.env.SMMCOST_API_KEY || '91e5a8595f63688b6c3bd59fa80218de';
  private static readonly BASE_URL = 'https://smmcost.com/api/v2';

  static async getSingleOrderStatus(orderId: string): Promise<TubeBoostOrderStatus | { error: string }> {
    try {
      const response = await fetch(`${this.BASE_URL}`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          key: this.API_KEY,
          action: 'status',
          order: orderId
        })
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const data = await response.json();
      return data;
    } catch (error) {
      console.error('Error fetching order status:', error);
      return { error: 'Failed to fetch status' };
    }
  }

  static async getOrdersStatus(orderIds: string[]): Promise<TubeBoostStatusResponse> {
    const results: TubeBoostStatusResponse = {};
    
    // Process orders in batches to avoid overwhelming the API
    for (const orderId of orderIds) {
      results[orderId] = await this.getSingleOrderStatus(orderId);
      // Small delay between requests to be respectful to the API
      await new Promise(resolve => setTimeout(resolve, 100));
    }
    
    return results;
  }

  // Status values from TubeBoost API:
  // "Pending" - order is being processed
  // "In progress" - order is running
  // "Partial" - order is partially completed
  // "Completed" - order is fully completed
  // "Canceled" - order was canceled
  // "Processing" - order is being processed
  static isOrderCompleted(status: string): boolean {
    return status === 'Completed';
  }

  static isOrderActive(status: string): boolean {
    return ['Pending', 'In progress', 'Partial', 'Processing'].includes(status);
  }

  static isOrderFailed(status: string): boolean {
    return status === 'Canceled';
  }
}