/**
 * Test script to verify ANTHROPIC_API_KEY_TUBEBOOST configuration
 */

import Anthropic from '@anthropic-ai/sdk';

// Test the API key configuration
async function testAnthropicKeys() {
  console.log('🔑 Testing Anthropic API key configurations...\n');

  // Check environment variables
  const primaryKey = process.env.ANTHROPIC_API_KEY_TUBEBOOST;
  const fallbackKey = process.env.ANTHROPIC_API_KEY;

  console.log('Environment Variables Status:');
  console.log(`- ANTHROPIC_API_KEY_TUBEBOOST: ${primaryKey ? '✅ Set' : '❌ Not set'}`);
  console.log(`- ANTHROPIC_API_KEY: ${fallbackKey ? '✅ Set' : '❌ Not set'}\n`);

  if (!primaryKey && !fallbackKey) {
    console.error('❌ No Anthropic API keys found in environment variables');
    return;
  }

  // Test ANTHROPIC_API_KEY_TUBEBOOST first (priority key)
  if (primaryKey) {
    console.log('🧪 Testing ANTHROPIC_API_KEY_TUBEBOOST...');
    const anthropicPrimary = new Anthropic({
      apiKey: primaryKey,
    });

    try {
      const response = await anthropicPrimary.messages.create({
        model: "claude-3-5-sonnet-20241022",
        max_tokens: 50,
        messages: [
          {
            role: "user",
            content: "Test message: Reply with 'TUBEBOOST KEY WORKING' if you receive this."
          }
        ]
      });

      const content = response.content[0];
      if (content.type === 'text') {
        console.log(`✅ ANTHROPIC_API_KEY_TUBEBOOST: ${content.text}`);
      }
    } catch (error) {
      console.error(`❌ ANTHROPIC_API_KEY_TUBEBOOST failed: ${error instanceof Error ? error.message : 'Unknown error'}`);
    }
  }

  // Test fallback key
  if (fallbackKey) {
    console.log('\n🧪 Testing ANTHROPIC_API_KEY...');
    const anthropicFallback = new Anthropic({
      apiKey: fallbackKey,
    });

    try {
      const response = await anthropicFallback.messages.create({
        model: "claude-3-5-sonnet-20241022",
        max_tokens: 50,
        messages: [
          {
            role: "user",
            content: "Test message: Reply with 'FALLBACK KEY WORKING' if you receive this."
          }
        ]
      });

      const content = response.content[0];
      if (content.type === 'text') {
        console.log(`✅ ANTHROPIC_API_KEY: ${content.text}`);
      }
    } catch (error) {
      console.error(`❌ ANTHROPIC_API_KEY failed: ${error instanceof Error ? error.message : 'Unknown error'}`);
    }
  }

  console.log('\n🏁 Test completed');
}

// Run the test
if (import.meta.url === `file://${process.argv[1]}`) {
  testAnthropicKeys().catch(console.error);
}

export { testAnthropicKeys };