export interface YouTubeVideoStats {
  videoId: string;
  // Statistics
  likeCount: string;
  viewCount: string;
  commentCount: string;
  favoriteCount: string;
  // Snippet
  publishedAt: string;
  channelId: string;
  channelTitle: string;
  title: string;
  description: string;
  tags: string[];
  categoryId: string;
  liveBroadcastContent: string;
  defaultLanguage: string;
  defaultAudioLanguage: string;
  // Content Details
  duration: string;
  durationSeconds: number;
  dimension: string;
  definition: string;
  caption: string;
  licensedContent: boolean;
  projection: string;
  // Status
  uploadStatus: string;
  privacyStatus: string;
  license: string;
  embeddable: boolean;
  publicStatsViewable: boolean;
  madeForKids: boolean;
  // Recording Details
  recordingDate: string;
  // Topic Details
  topicCategories: string[];
  // Thumbnails
  thumbnails: {
    default?: { url: string; width: number; height: number };
    medium?: { url: string; width: number; height: number };
    high?: { url: string; width: number; height: number };
    standard?: { url: string; width: number; height: number };
    maxres?: { url: string; width: number; height: number };
  };
}

export class YouTubeAPIService {
  private static readonly API_KEY = process.env.YOUTUBE_API_KEY;
  private static readonly BASE_URL = 'https://www.googleapis.com/youtube/v3';

  // Helper function to convert YouTube ISO 8601 duration to seconds
  private static parseDuration(duration: string): number {
    const match = duration.match(/PT(?:(\d+)H)?(?:(\d+)M)?(?:(\d+)S)?/);
    if (!match) return 0;
    
    const hours = parseInt(match[1]) || 0;
    const minutes = parseInt(match[2]) || 0;
    const seconds = parseInt(match[3]) || 0;
    
    return hours * 3600 + minutes * 60 + seconds;
  }

  // Helper function to format duration seconds to readable format
  private static formatDuration(seconds: number): string {
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = seconds % 60;
    
    if (hours > 0) {
      return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    } else {
      return `${minutes}:${secs.toString().padStart(2, '0')}`;
    }
  }

  static async getVideoStats(videoIds: string[]): Promise<YouTubeVideoStats[]> {
    if (!this.API_KEY) {
      throw new Error('YouTube API key not configured');
    }

    if (videoIds.length === 0) {
      return [];
    }

    try {
      const videoIdsString = videoIds.join(',');
      const url = `${this.BASE_URL}/videos?part=statistics,snippet,contentDetails,status,recordingDetails,topicDetails&id=${videoIdsString}&key=${this.API_KEY}`;
      
      const response = await fetch(url);
      const data = await response.json();

      if (data.error) {
        console.error('YouTube API error:', data.error);
        throw new Error(`YouTube API error: ${data.error.message}`);
      }

      return data.items?.map((item: any) => {
        const durationSeconds = this.parseDuration(item.contentDetails?.duration || 'PT0S');
        return {
          videoId: item.id,
          // Statistics
          likeCount: item.statistics?.likeCount || '0',
          viewCount: item.statistics?.viewCount || '0',
          commentCount: item.statistics?.commentCount || '0',
          favoriteCount: item.statistics?.favoriteCount || '0',
          // Snippet
          publishedAt: item.snippet?.publishedAt || '',
          channelId: item.snippet?.channelId || '',
          channelTitle: item.snippet?.channelTitle || '',
          title: item.snippet?.title || '',
          description: item.snippet?.description || '',
          tags: item.snippet?.tags || [],
          categoryId: item.snippet?.categoryId || '',
          liveBroadcastContent: item.snippet?.liveBroadcastContent || 'none',
          defaultLanguage: item.snippet?.defaultLanguage || '',
          defaultAudioLanguage: item.snippet?.defaultAudioLanguage || '',
          // Content Details
          duration: this.formatDuration(durationSeconds),
          durationSeconds,
          dimension: item.contentDetails?.dimension || '2d',
          definition: item.contentDetails?.definition || 'sd',
          caption: item.contentDetails?.caption || 'false',
          licensedContent: item.contentDetails?.licensedContent || false,
          projection: item.contentDetails?.projection || 'rectangular',
          // Status
          uploadStatus: item.status?.uploadStatus || 'processed',
          privacyStatus: item.status?.privacyStatus || 'public',
          license: item.status?.license || 'youtube',
          embeddable: item.status?.embeddable !== false,
          publicStatsViewable: item.status?.publicStatsViewable !== false,
          madeForKids: item.status?.madeForKids || false,
          // Recording Details
          recordingDate: item.recordingDetails?.recordingDate || '',
          // Topic Details
          topicCategories: item.topicDetails?.topicCategories || [],
          // Thumbnails
          thumbnails: item.snippet?.thumbnails || {}
        };
      }) || [];
    } catch (error) {
      console.error('Error fetching YouTube video stats:', error);
      throw error;
    }
  }

  static async getSingleVideoStats(videoId: string): Promise<YouTubeVideoStats | null> {
    const stats = await this.getVideoStats([videoId]);
    return stats[0] || null;
  }
}