import { pgTable, text, serial, integer, boolean, timestamp, real, unique, decimal, json } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

export const users = pgTable("users", {
  id: serial("id").primaryKey(),
  username: text("username").notNull().unique(),
  email: text("email").notNull().unique(),
  password: text("password").notNull(),
  firstName: text("first_name").notNull(),
  lastName: text("last_name").notNull(),
  mobileNumber: text("mobile_number").notNull(),
  country: text("country").notNull(),
  city: text("city"),
  company: text("company"),
  jobTitle: text("job_title"),
  website: text("website"),
  accountType: text("account_type").notNull().default("regular"), // Only 'regular' now, no anonymous
  subscriptionTier: text("subscription_tier").notNull().default("free"), // 'free', 'starter', 'professional', 'enterprise'
  role: text("role").notNull().default("client"), // 'admin', 'client'
  isActive: boolean("is_active").notNull().default(true),
  purchasedPoints: integer("purchased_points").notNull().default(0), // Never expire
  lastLogin: timestamp("last_login"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const sessions = pgTable("sessions", {
  id: text("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  expiresAt: timestamp("expires_at").notNull(),
  createdAt: timestamp("created_at").defaultNow(),
});

export const channels = pgTable("channels", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull(), // Original creator
  channelId: text("channel_id").notNull().unique(),
  channelName: text("channel_name").notNull(),
  channelHandle: text("channel_handle"), // YouTube @handle for proper URL generation
  customUrl: text("custom_url"), // YouTube custom URL if available
  agScore: real("ag_score").notNull().default(7.8),
  lastReplenishTime: timestamp("last_replenish_time"),
  totalViews: integer("total_views").notNull().default(0),
  subscribers: integer("subscribers").notNull().default(0),
  watchTimeHours: integer("watch_time_hours").notNull().default(0),
  isActive: boolean("is_active").notNull().default(false),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Junction table for many-to-many relationship between users and channels
export const userChannels = pgTable("user_channels", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id),
  channelId: integer("channel_id").notNull().references(() => channels.id),
  isVisible: boolean("is_visible").notNull().default(true), // For client-level soft deletion
  isCurrentActive: boolean("is_current_active").notNull().default(false), // Currently selected channel for user
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => ({
  userChannelUnique: unique("user_channel_unique").on(table.userId, table.channelId),
}));

export const videos = pgTable("videos", {
  id: serial("id").primaryKey(),
  channelId: integer("channel_id").notNull(),
  videoId: text("video_id").notNull().unique(),
  title: text("title").notNull(),
  description: text("description"),
  thumbnailUrl: text("thumbnail_url"),
  views: integer("views").notNull().default(0),
  duration: text("duration"),
  publishedAt: timestamp("published_at"),
  algorithmScore: real("algorithm_score").notNull().default(7.5),
  ctr: real("ctr").notNull().default(0.0),
  engagementRate: real("engagement_rate").notNull().default(0.0),
  seoScore: real("seo_score").notNull().default(0.0),
  lastBoosted: timestamp("last_boosted"),
  lastSessionCreated: timestamp("last_session_created"),
  youtubeAdvice: text("youtube_advice"),
  adviceGeneratedAt: timestamp("advice_generated_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const agSessions = pgTable("ag_sessions", {
  id: serial("id").primaryKey(),
  channelId: integer("channel_id").notNull(),
  videoId: integer("video_id"),
  sessionType: text("session_type").notNull(), // 'boost', 'session', 'replenish', 'devalue', 'whatsapp'
  duration: integer("duration").notNull(), // in seconds
  status: text("status").notNull().default('pending'), // 'pending', 'running', 'completed', 'failed'
  progress: real("progress").notNull().default(0.0),
  startedAt: timestamp("started_at").defaultNow(),
  completedAt: timestamp("completed_at"),
});

export const videoOrders = pgTable("video_orders", {
  id: serial("id").primaryKey(),
  videoId: integer("video_id").notNull().references(() => videos.id),
  channelId: integer("channel_id").notNull().references(() => channels.id),
  orderId: text("order_id").notNull(),
  service: text("service").notNull().default("1523"),
  quantity: integer("quantity").notNull(),
  status: text("status"),
  charge: text("charge"),
  startCount: text("start_count"),
  remains: text("remains"),
  currency: text("currency"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const channelViewsLimits = pgTable("channel_views_limits", {
  id: serial("id").primaryKey(),
  channelId: integer("channel_id").notNull().references(() => channels.id).unique(),
  usedCount: integer("used_count").default(0),
  lastResetTime: timestamp("last_reset_time").defaultNow(),
});

export const refillOrders = pgTable("refill_orders", {
  id: serial("id").primaryKey(),
  originalOrderId: integer("original_order_id").notNull().references(() => videoOrders.id),
  refillId: text("refill_id").notNull(),
  status: text("status").default("pending"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const subscriberOrders = pgTable("subscriber_orders", {
  id: serial("id").primaryKey(),
  channelId: integer("channel_id").notNull().references(() => channels.id, { onDelete: 'cascade' }),
  orderId: text("order_id").notNull(),
  service: text("service").notNull().default("1076"),
  quantity: integer("quantity").notNull(),
  channelUrl: text("channel_url").notNull(),
  status: text("status").default("Pending"),
  charge: text("charge"),
  startCount: text("start_count"),
  remains: text("remains"),
  currency: text("currency").default("USD"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const subscriberRefillOrders = pgTable("subscriber_refill_orders", {
  id: serial("id").primaryKey(),
  originalOrderId: integer("original_order_id").notNull().references(() => subscriberOrders.id),
  refillId: text("refill_id").notNull(),
  status: text("status").default("pending"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// System settings for point allocation
export const systemSettings = pgTable("system_settings", {
  id: serial("id").primaryKey(),
  key: text("key").notNull().unique(),
  value: text("value").notNull(),
  description: text("description"),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// User point allocations - tracks daily point assignments
export const userPointAllocations = pgTable("user_point_allocations", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  date: text("date").notNull(), // YYYY-MM-DD format
  totalPoints: integer("total_points").notNull().default(600),
  usedPoints: integer("used_points").notNull().default(0),
  remainingPoints: integer("remaining_points").notNull().default(600),
  isCustomAllocation: boolean("is_custom_allocation").notNull().default(false), // true if admin set custom points
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  userDateUnique: unique("user_date_unique").on(table.userId, table.date),
}));

// Point transactions - logs all point usage
export const pointTransactions = pgTable("point_transactions", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  allocationId: integer("allocation_id").notNull().references(() => userPointAllocations.id, { onDelete: "cascade" }),
  points: integer("points").notNull(), // negative for deductions, positive for additions
  type: text("type").notNull(), // 'video_order', 'subscriber_order', 'refill', 'admin_adjustment', etc.
  description: text("description"),
  relatedId: integer("related_id"), // ID of related order/action
  createdAt: timestamp("created_at").defaultNow(),
});

// Point purchases - Stripe payment tracking for purchased points (never expire)
export const pointPurchases = pgTable("point_purchases", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  stripePaymentIntentId: text("stripe_payment_intent_id").notNull().unique(),
  points: integer("points").notNull(),
  amountCents: integer("amount_cents").notNull(),
  currency: text("currency").notNull().default("USD"),
  status: text("status").notNull().default("pending"), // 'pending', 'completed', 'failed'
  createdAt: timestamp("created_at").defaultNow(),
  completedAt: timestamp("completed_at"),
});

// Subscription packages management
export const subscriptionPackages = pgTable("subscription_packages", {
  id: serial("id").primaryKey(),
  name: text("name").notNull().unique(), // 'Free', 'Starter', 'Professional', 'Enterprise'
  displayName: text("display_name").notNull(),
  description: text("description"),
  price: decimal("price", { precision: 10, scale: 2 }).notNull().default("0.00"),
  currency: text("currency").notNull().default("USD"),
  billingCycle: text("billing_cycle").notNull().default("monthly"), // 'monthly', 'yearly'
  dailyPoints: integer("daily_points").notNull().default(600),
  weeklyPoints: integer("weekly_points").notNull().default(4200), // 7 * daily
  monthlyPoints: integer("monthly_points").notNull().default(18000), // 30 * daily
  pointAllocationMethod: text("point_allocation_method").notNull().default("daily"), // 'daily', 'weekly', 'monthly'
  maxChannels: integer("max_channels").notNull().default(1),
  maxVideosPerDay: integer("max_videos_per_day").notNull().default(10),
  features: json("features").default([]), // Array of feature names
  isActive: boolean("is_active").notNull().default(true),
  sortOrder: integer("sort_order").notNull().default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// User package assignments
export const userPackageAssignments = pgTable("user_package_assignments", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  packageId: integer("package_id").notNull().references(() => subscriptionPackages.id, { onDelete: "cascade" }),
  assignedAt: timestamp("assigned_at").defaultNow(),
  assignedBy: integer("assigned_by").references(() => users.id), // Admin who assigned it
  isActive: boolean("is_active").notNull().default(true),
  expiresAt: timestamp("expires_at"), // For time-limited assignments
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => ({
  userPackageUnique: unique("user_package_unique").on(table.userId, table.packageId),
}));

// Header Code Management - for Facebook/Google pixels and tracking codes
export const headerCodes = pgTable("header_codes", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(), // "Facebook Pixel", "Google Analytics", etc.
  code: text("code").notNull(), // The actual HTML/JavaScript code
  isActive: boolean("is_active").notNull().default(true),
  createdBy: integer("created_by").notNull().references(() => users.id),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// AI Configuration for moderation system
export const aiConfigurations = pgTable("ai_configurations", {
  id: serial("id").primaryKey(),
  name: text("name").notNull().unique(), // "Default Configuration", "Reddit Moderation", etc.
  provider: text("provider").notNull(), // 'anthropic', 'openai', 'azure'
  model: text("model").notNull(), // 'claude-sonnet-4', 'gpt-4o', etc.
  apiKey: text("api_key"), // Provider-specific API key
  systemPrompt: text("system_prompt").notNull().default("You are an AI Moderator for Reddit and are moderating Subreddits to ensure people follow the rules."),
  apiEndpoint: text("api_endpoint"), // For Azure custom endpoints - auto-managed
  maxTokens: integer("max_tokens").notNull().default(1000),
  temperature: real("temperature").notNull().default(0.3),
  topP: real("top_p").notNull().default(0.1),
  isActive: boolean("is_active").notNull().default(false), // Only one can be active
  isDefault: boolean("is_default").notNull().default(false), // Default for new configurations
  createdBy: integer("created_by").notNull().references(() => users.id),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// AI Error Logs for advanced monitoring
export const aiErrorLogs = pgTable("ai_error_logs", {
  id: serial("id").primaryKey(),
  configurationId: integer("configuration_id").references(() => aiConfigurations.id),
  provider: text("provider").notNull(),
  model: text("model").notNull(),
  endpoint: text("endpoint"),
  requestPayload: json("request_payload"),
  responseData: json("response_data"),
  errorMessage: text("error_message").notNull(),
  errorCode: text("error_code"),
  httpStatus: integer("http_status"),
  requestDuration: integer("request_duration"), // in milliseconds
  retryAttempt: integer("retry_attempt").notNull().default(0),
  contextData: json("context_data"), // Additional context (user ID, post ID, etc.)
  severity: text("severity").notNull().default('error'), // 'warning', 'error', 'critical'
  isResolved: boolean("is_resolved").notNull().default(false),
  resolvedBy: integer("resolved_by").references(() => users.id),
  resolvedAt: timestamp("resolved_at"),
  createdAt: timestamp("created_at").defaultNow(),
});

// AI Usage Analytics
export const aiUsageStats = pgTable("ai_usage_stats", {
  id: serial("id").primaryKey(),
  configurationId: integer("configuration_id").notNull().references(() => aiConfigurations.id),
  provider: text("provider").notNull(),
  model: text("model").notNull(),
  requestCount: integer("request_count").notNull().default(0),
  successCount: integer("success_count").notNull().default(0),
  errorCount: integer("error_count").notNull().default(0),
  totalTokensUsed: integer("total_tokens_used").notNull().default(0),
  averageResponseTime: real("average_response_time").notNull().default(0),
  dateRecord: timestamp("date_record").notNull(), // Daily aggregation
  createdAt: timestamp("created_at").defaultNow(),
});



export const insertUserSchema = createInsertSchema(users).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  lastLogin: true,
});

export const insertSessionSchema = createInsertSchema(sessions).omit({
  createdAt: true,
});

export const insertChannelSchema = createInsertSchema(channels).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertUserChannelSchema = createInsertSchema(userChannels).omit({
  id: true,
  createdAt: true,
});

export const insertVideoSchema = createInsertSchema(videos).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertAgSessionSchema = createInsertSchema(agSessions).omit({
  id: true,
  startedAt: true,
  completedAt: true,
});

export const insertVideoOrderSchema = createInsertSchema(videoOrders).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertChannelViewsLimitSchema = createInsertSchema(channelViewsLimits).omit({
  id: true,
  lastResetTime: true,
});

export const insertRefillOrderSchema = createInsertSchema(refillOrders).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertSubscriberOrderSchema = createInsertSchema(subscriberOrders).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertSubscriberRefillOrderSchema = createInsertSchema(subscriberRefillOrders).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertSystemSettingSchema = createInsertSchema(systemSettings).omit({
  id: true,
  updatedAt: true,
});

export const insertUserPointAllocationSchema = createInsertSchema(userPointAllocations).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertPointTransactionSchema = createInsertSchema(pointTransactions).omit({
  id: true,
  createdAt: true,
});

export const insertPointPurchaseSchema = createInsertSchema(pointPurchases).omit({
  id: true,
  createdAt: true,
  completedAt: true,
});

export const insertSubscriptionPackageSchema = createInsertSchema(subscriptionPackages).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertUserPackageAssignmentSchema = createInsertSchema(userPackageAssignments).omit({
  id: true,
  createdAt: true,
});

// Queue system for managing multiple simulations across channels
export const queueItems = pgTable("queue_items", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id),
  channelId: integer("channel_id").notNull().references(() => channels.id),
  actionType: text("action_type").notNull(), // 'boost', 'session', 'replenish', 'devalue', 'whatsapp'
  status: text("status").notNull().default("queued"), // 'queued', 'running', 'completed', 'failed', 'cancelled'
  videoId: integer("video_id").references(() => videos.id), // For video-specific actions
  sessionId: integer("session_id").references(() => agSessions.id), // Link to actual session when started
  priority: integer("priority").notNull().default(0), // Higher numbers = higher priority
  progress: integer("progress").notNull().default(0), // 0-100
  currentStep: text("current_step"),
  estimatedDuration: integer("estimated_duration"), // in seconds
  startedAt: timestamp("started_at"),
  completedAt: timestamp("completed_at"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertQueueItemSchema = createInsertSchema(queueItems).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertUser = z.infer<typeof insertUserSchema>;
export type User = typeof users.$inferSelect;
export type InsertSession = z.infer<typeof insertSessionSchema>;
export type Session = typeof sessions.$inferSelect;
export type InsertChannel = z.infer<typeof insertChannelSchema>;
export type Channel = typeof channels.$inferSelect;
export type InsertUserChannel = z.infer<typeof insertUserChannelSchema>;
export type UserChannel = typeof userChannels.$inferSelect;
export type InsertVideo = z.infer<typeof insertVideoSchema>;
export type Video = typeof videos.$inferSelect;
export type InsertAgSession = z.infer<typeof insertAgSessionSchema>;
export type AgSession = typeof agSessions.$inferSelect;
export type InsertVideoOrder = z.infer<typeof insertVideoOrderSchema>;
export type VideoOrder = typeof videoOrders.$inferSelect;
export type InsertChannelViewsLimit = z.infer<typeof insertChannelViewsLimitSchema>;
export type ChannelViewsLimit = typeof channelViewsLimits.$inferSelect;
export type InsertRefillOrder = z.infer<typeof insertRefillOrderSchema>;
export type RefillOrder = typeof refillOrders.$inferSelect;
export type InsertSubscriberOrder = z.infer<typeof insertSubscriberOrderSchema>;
export type SubscriberOrder = typeof subscriberOrders.$inferSelect;
export type InsertSubscriberRefillOrder = z.infer<typeof insertSubscriberRefillOrderSchema>;
export type SubscriberRefillOrder = typeof subscriberRefillOrders.$inferSelect;
export type InsertSystemSetting = z.infer<typeof insertSystemSettingSchema>;
export type SystemSetting = typeof systemSettings.$inferSelect;
export type InsertUserPointAllocation = z.infer<typeof insertUserPointAllocationSchema>;
export type UserPointAllocation = typeof userPointAllocations.$inferSelect;
export type InsertPointTransaction = z.infer<typeof insertPointTransactionSchema>;
export type PointTransaction = typeof pointTransactions.$inferSelect;
export type InsertPointPurchase = z.infer<typeof insertPointPurchaseSchema>;
export type PointPurchase = typeof pointPurchases.$inferSelect;
export type InsertQueueItem = z.infer<typeof insertQueueItemSchema>;
export type QueueItem = typeof queueItems.$inferSelect;
export type InsertSubscriptionPackage = z.infer<typeof insertSubscriptionPackageSchema>;
export type SubscriptionPackage = typeof subscriptionPackages.$inferSelect;
export type InsertUserPackageAssignment = z.infer<typeof insertUserPackageAssignmentSchema>;
export type UserPackageAssignment = typeof userPackageAssignments.$inferSelect;



// Device fingerprinting and security tracking
export const deviceFingerprints = pgTable("device_fingerprints", {
  id: serial("id").primaryKey(),
  userId: text("user_id").notNull(), // Changed to text to match database
  ipAddress: text("ip_address").notNull(),
  fingerprintHash: text("fingerprint_hash").notNull().unique(), // Match database column name
  userAgent: text("user_agent").notNull(),
  screenResolution: text("screen_resolution"),
  timezone: text("timezone"),
  language: text("language"),
  platform: text("platform"),
  webglRenderer: text("webgl_renderer"),
  canvasFingerprint: text("canvas_fingerprint"),
  audioFingerprint: text("audio_fingerprint"),
  macAddress: text("mac_address"), // When available via WebRTC
  networkInterfaces: text("network_interfaces"), // JSON array of network info
  hardwareFingerprint: text("hardware_fingerprint"), // CPU cores, memory, etc.
  browserFingerprint: text("browser_fingerprint"), // Plugins, extensions, etc.
  isBlocked: boolean("is_blocked").notNull().default(false),
  blockedReason: text("blocked_reason"), // Match database column name
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// IP address tracking for household detection
export const ipAddressLogs = pgTable("ip_address_logs", {
  id: serial("id").primaryKey(),
  ipAddress: text("ip_address").notNull(),
  ipHash: text("ip_hash").notNull().unique(), // Match database column name
  country: text("country"),
  region: text("region"),
  city: text("city"),
  isp: text("isp"),
  isVpn: boolean("is_vpn").default(false),
  isProxy: boolean("is_proxy").default(false),
  userCount: integer("user_count").notNull().default(1),
  lastSeen: timestamp("last_seen").defaultNow(),
});

// Security violations and blocking
export const securityViolations = pgTable("security_violations", {
  id: serial("id").primaryKey(),
  userId: integer("userId").references(() => users.id, { onDelete: "cascade" }),
  fingerprintId: integer("fingerprintId").references(() => deviceFingerprints.id, { onDelete: "cascade" }),
  violationType: text("violationType").notNull(), // 'duplicate_household', 'vpn_detected', 'suspicious_device'
  description: text("description").notNull(),
  severity: text("severity").notNull().default("medium"), // 'low', 'medium', 'high', 'critical'
  actionTaken: text("actionTaken"), // 'blocked', 'flagged', 'logged'
  ipAddress: text("ipAddress"),
  userAgent: text("userAgent"),
  additionalData: text("additionalData"), // JSON for extra context
  createdAt: timestamp("createdAt").defaultNow(),
});

export const insertDeviceFingerprintSchema = createInsertSchema(deviceFingerprints).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertIpAddressLogSchema = createInsertSchema(ipAddressLogs).omit({
  id: true,
  lastSeen: true,
});

export const insertSecurityViolationSchema = createInsertSchema(securityViolations).omit({
  id: true,
  createdAt: true,
});

export type InsertDeviceFingerprint = z.infer<typeof insertDeviceFingerprintSchema>;
export type DeviceFingerprint = typeof deviceFingerprints.$inferSelect;
export type InsertIpAddressLog = z.infer<typeof insertIpAddressLogSchema>;
export type IpAddressLog = typeof ipAddressLogs.$inferSelect;
export type InsertSecurityViolation = z.infer<typeof insertSecurityViolationSchema>;
export type SecurityViolation = typeof securityViolations.$inferSelect;

// Reddit connections table
export const redditConnections = pgTable("reddit_connections", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").references(() => users.id, { onDelete: "cascade" }).notNull(),
  redditUserId: text("reddit_user_id").notNull(),
  redditUsername: text("reddit_username").notNull(),
  accessToken: text("access_token").notNull(),
  refreshToken: text("refresh_token").notNull(),
  expiresAt: timestamp("expires_at").notNull(),
  scope: text("scope").notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  userIdUnique: unique("user_id_unique").on(table.userId),
}));

export const insertRedditConnectionSchema = createInsertSchema(redditConnections).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertRedditConnection = z.infer<typeof insertRedditConnectionSchema>;
export type RedditConnection = typeof redditConnections.$inferSelect;

// Reddit Posts table for post scheduling and management
export const redditPosts = pgTable("reddit_posts", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  title: text("title").notNull(),
  content: text("content").notNull(),
  subreddit: text("subreddit").notNull(),
  status: text("status").notNull().default("draft"), // draft, scheduled, published, failed
  scheduledAt: timestamp("scheduled_at"),
  publishedAt: timestamp("published_at"),
  redditPostId: text("reddit_post_id"),
  upvotes: integer("upvotes").default(0),
  comments: integer("comments").default(0),
  engagement: real("engagement").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});

// Subreddit Analytics table
export const subredditAnalytics = pgTable("subreddit_analytics", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  subredditName: text("subreddit_name").notNull(),
  subscribers: integer("subscribers").default(0),
  activeUsers: integer("active_users").default(0),
  growthRate: real("growth_rate").default(0),
  engagementRate: real("engagement_rate").default(0),
  bestPostTimes: text("best_post_times").array(),
  topKeywords: text("top_keywords").array(),
  competitorPosts: integer("competitor_posts").default(0),
  lastAnalyzed: timestamp("last_analyzed").defaultNow(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});

export const insertRedditPostSchema = createInsertSchema(redditPosts).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertSubredditAnalyticsSchema = createInsertSchema(subredditAnalytics).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const managedSubreddits = pgTable("managed_subreddits", {
  id: serial("id").primaryKey(),
  userId: text("user_id").notNull(),
  subredditName: text("subreddit_name").notNull(),
  displayName: text("display_name").notNull(),
  isOwner: boolean("is_owner").default(false),
  isModerator: boolean("is_moderator").default(false),
  permissions: text("permissions").array().default([]),
  subscribers: integer("subscribers").default(0),
  activeUsers: integer("active_users").default(0),
  description: text("description"),
  isNsfw: boolean("is_nsfw").default(false),
  subredditType: text("subreddit_type").default("public"),
  addedAt: timestamp("added_at").defaultNow(),
  lastUpdated: timestamp("last_updated").defaultNow()
});

export const subredditAnalyticsExtended = pgTable("subreddit_analytics_extended", {
  id: serial("id").primaryKey(),
  managedSubredditId: integer("managed_subreddit_id").notNull(),
  avgUpvotes: integer("avg_upvotes").default(0),
  avgComments: integer("avg_comments").default(0),
  topPostTime: text("top_post_time").default("12:00 PM"),
  contentCategories: text("content_categories").array().default([]),
  moderatorCount: integer("moderator_count").default(0),
  weeklyPosts: integer("weekly_posts").default(0),
  monthlyGrowth: decimal("monthly_growth", { precision: 5, scale: 2 }).default("0"),
  engagementTrend: text("engagement_trend").default("stable"),
  peakHours: text("peak_hours").array().default([]),
  topContributors: text("top_contributors").array().default([]),
  lastAnalyzed: timestamp("last_analyzed").defaultNow(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow()
});

export const insertManagedSubredditSchema = createInsertSchema(managedSubreddits).omit({
  id: true,
  addedAt: true,
  lastUpdated: true,
});

export const insertSubredditAnalyticsExtendedSchema = createInsertSchema(subredditAnalyticsExtended).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type RedditPost = typeof redditPosts.$inferSelect;
export type InsertRedditPost = z.infer<typeof insertRedditPostSchema>;
export type SubredditAnalytics = typeof subredditAnalytics.$inferSelect;
export type InsertSubredditAnalytics = z.infer<typeof insertSubredditAnalyticsSchema>;
export type ManagedSubreddit = typeof managedSubreddits.$inferSelect;
export type InsertManagedSubreddit = z.infer<typeof insertManagedSubredditSchema>;
export type SubredditAnalyticsExtended = typeof subredditAnalyticsExtended.$inferSelect;
export type InsertSubredditAnalyticsExtended = z.infer<typeof insertSubredditAnalyticsExtendedSchema>;

// Reddit Moderation Tables
export const moderationQueueItems = pgTable("moderation_queue_items", {
  id: serial("id").primaryKey(),
  redditId: text("reddit_id").notNull().unique(),
  subredditName: text("subreddit_name").notNull(),
  itemType: text("item_type").notNull(), // 'submission' or 'comment'
  author: text("author").notNull(),
  title: text("title"),
  content: text("content").notNull(),
  url: text("url"),
  permalink: text("permalink").notNull(),
  score: integer("score").default(0),
  numReports: integer("num_reports").default(0),
  userReports: text("user_reports").default('[]'), // JSON string
  modReports: text("mod_reports").default('[]'), // JSON string
  createdUtc: integer("created_utc").notNull(),
  processed: boolean("processed").default(false),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const aiAnalysisResults = pgTable("ai_analysis_results", {
  id: serial("id").primaryKey(),
  modqueueItemId: integer("modqueue_item_id").notNull().references(() => moderationQueueItems.id, { onDelete: "cascade" }),
  violationsFound: text("violations_found").default('[]'), // JSON string
  suggestedAction: text("suggested_action").notNull(), // 'approve', 'remove', 'spam', 'manual_review'
  reasoning: text("reasoning").notNull(),
  confidenceScore: decimal("confidence_score", { precision: 3, scale: 2 }).notNull(),
  severity: text("severity").notNull().default('medium'), // 'low', 'medium', 'high'
  removalReason: text("removal_reason"),
  requiresHumanReview: boolean("requires_human_review").default(false),
  additionalActions: text("additional_actions").default('[]'), // JSON string
  createdAt: timestamp("created_at").defaultNow(),
});

export const moderationActions = pgTable("moderation_actions", {
  id: serial("id").primaryKey(),
  modqueueItemId: integer("modqueue_item_id").references(() => moderationQueueItems.id, { onDelete: "cascade" }),
  userId: integer("user_id").references(() => users.id, { onDelete: "cascade" }),
  actionType: text("action_type").notNull(), // 'approve', 'remove', 'spam', 'lock', 'sticky'
  actionReason: text("action_reason"),
  wasAiSuggested: boolean("was_ai_suggested").default(false),
  aiConfidenceScore: decimal("ai_confidence_score", { precision: 3, scale: 2 }),
  moderatorNote: text("moderator_note"),
  humanResponse: text("human_response"), // Human-like response for modmail
  actionTimestamp: timestamp("action_timestamp").defaultNow(),
  redditResponse: json("reddit_response"),
});

export const subredditRulesCache = pgTable("subreddit_rules_cache", {
  id: serial("id").primaryKey(),
  subredditName: text("subreddit_name").notNull().unique(),
  rulesJson: text("rules_json").notNull(), // JSON string
  description: text("description"),
  sidebarHtml: text("sidebar_html"),
  subredditType: text("subreddit_type"),
  submissionType: text("submission_type"),
  lastUpdated: timestamp("last_updated").defaultNow(),
  createdAt: timestamp("created_at").defaultNow(),
});

// Insert schemas for moderation tables
export const insertModerationQueueItemSchema = createInsertSchema(moderationQueueItems).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertAiAnalysisResultSchema = createInsertSchema(aiAnalysisResults).omit({
  id: true,
  createdAt: true,
});

export const insertModerationActionSchema = createInsertSchema(moderationActions).omit({
  id: true,
  actionTimestamp: true,
}).extend({
  modqueueItemId: z.number().nullable(),
  userId: z.number().optional(),
});

export const insertSubredditRulesCacheSchema = createInsertSchema(subredditRulesCache).omit({
  id: true,
  createdAt: true,
});

// Types for moderation tables
export type InsertModerationQueueItem = z.infer<typeof insertModerationQueueItemSchema>;
export type ModerationQueueItem = typeof moderationQueueItems.$inferSelect;
export type InsertAiAnalysisResult = z.infer<typeof insertAiAnalysisResultSchema>;
export type AiAnalysisResult = typeof aiAnalysisResults.$inferSelect;
export type InsertModerationAction = z.infer<typeof insertModerationActionSchema>;
export type ModerationAction = typeof moderationActions.$inferSelect;
export type InsertSubredditRulesCache = z.infer<typeof insertSubredditRulesCacheSchema>;
export type SubredditRulesCache = typeof subredditRulesCache.$inferSelect;

// Redupvotes API Orders table for tracking upvote purchases
export const redupvotesOrders = pgTable("redupvotes_orders", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  redditPostId: text("reddit_post_id").notNull(), // The Reddit post ID (e.g., "1lw8by3")
  redditPostUrl: text("reddit_post_url").notNull(), // Full Reddit URL
  redupvotesOrderId: text("redupvotes_order_id").notNull(), // Order ID from Redupvotes API
  quantity: integer("quantity").notNull(), // Number of upvotes requested
  charge: decimal("charge", { precision: 10, scale: 5 }), // Cost in USD
  startCount: integer("start_count"), // Initial upvote count
  status: text("status").notNull().default("pending"), // Order status from API
  remains: integer("remains"), // Remaining upvotes to be delivered
  currency: text("currency").default("USD"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const insertRedupvotesOrderSchema = createInsertSchema(redupvotesOrders).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertHeaderCodeSchema = createInsertSchema(headerCodes).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

// AI Configuration insert schemas
export const insertAiConfigurationSchema = createInsertSchema(aiConfigurations).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertAiErrorLogSchema = createInsertSchema(aiErrorLogs).omit({
  id: true,
  createdAt: true,
});

export const insertAiUsageStatsSchema = createInsertSchema(aiUsageStats).omit({
  id: true,
  createdAt: true,
});

export type InsertRedupvotesOrder = z.infer<typeof insertRedupvotesOrderSchema>;
export type RedupvotesOrder = typeof redupvotesOrders.$inferSelect;
export type HeaderCode = typeof headerCodes.$inferSelect;
export type InsertHeaderCode = z.infer<typeof insertHeaderCodeSchema>;

// AI Configuration types
export type AiConfiguration = typeof aiConfigurations.$inferSelect;
export type InsertAiConfiguration = z.infer<typeof insertAiConfigurationSchema>;
export type AiErrorLog = typeof aiErrorLogs.$inferSelect;
export type InsertAiErrorLog = z.infer<typeof insertAiErrorLogSchema>;
export type AiUsageStats = typeof aiUsageStats.$inferSelect;
export type InsertAiUsageStats = z.infer<typeof insertAiUsageStatsSchema>;


